/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.workermanager.keepalive;

import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;

import org.eclipse.smila.taskmanager.Task;

/**
 * Delayed Task to keep tasks in a temporal ordering. Note: this class has a natural ordering that is inconsistent with
 * equals.
 */
public class DelayedTask implements Delayed {

  /** The task. */
  private final Task _task;

  /** the delay time (in nanoseconds). */
  private final long _delay;

  /**
   * The constructor of DelayedTask.
   * 
   * @param task
   *          the task.
   * @param delay
   *          the delay.
   * @param timeUnit
   *          the TineUnit for the delay.
   */
  public DelayedTask(final Task task, final long delay, final TimeUnit timeUnit) {
    _task = task;
    _delay = System.nanoTime() + TimeUnit.NANOSECONDS.convert(delay, timeUnit);
  }

  /**
   * {@inheritDoc} Note: this class has a natural ordering that is inconsistent with equals.
   */
  @Override
  public int compareTo(Delayed arg0) {
    return Long.valueOf(getDelay(TimeUnit.NANOSECONDS))
      .compareTo(Long.valueOf(arg0.getDelay(TimeUnit.NANOSECONDS)));
  }

  /**
   * {@inheritDoc} Note: this class has a natural ordering that is inconsistent with equals DelayedTasks are considered
   * equal when their task ID matches, no matter what the compareTo method would determine.
   */
  @Override
  public boolean equals(Object obj) {
    if (obj instanceof DelayedTask) {
      return (getTask().getTaskId().equals(((DelayedTask) obj).getTask().getTaskId()));
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return getTask().getTaskId().hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getDelay(TimeUnit unit) {
    return (unit.convert(_delay - System.nanoTime(), TimeUnit.NANOSECONDS));
  }

  /**
   * Returns the task.
   * 
   * @return the task
   */
  public Task getTask() {
    return _task;
  }
}
