/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.objectstore.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.ServiceUnavailableException;
import org.eclipse.smila.objectstore.StoreExistsException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.objectstore.StoreOutputStream;

/**
 * Utility class to retry objectstore operations.
 * 
 */
public final class ObjectStoreRetryUtil {

  /** How often should the generator retry to contact the server on IO error. */
  private static final int MAX_RETRY_ON_IOERROR = 3;

  /** How long should we wait between retries. */
  private static final int RETRY_WAIT = 10; // ms

  /** Logger. */
  private static final Log LOG = LogFactory.getLog(ObjectStoreRetryUtil.class);

  /**
   * private constructor (utility class!).
   */
  private ObjectStoreRetryUtil() {
    ;// to prevent construction
  }

  /**
   * Open a store and open an object for read. Retry on {@link ServiceUnavailableException}.
   * 
   * @param objectStore
   *          ObjectStoreService reference to use.
   * @param storeName
   *          the name of the store
   * @param objectId
   *          name of object to read.
   * @return an {@link InputStream} to read from the store.
   * @throws ObjectStoreException
   *           error reading object.
   */
  public static InputStream retryReadObject(final ObjectStoreService objectStore, final String storeName,
    final String objectId) throws ObjectStoreException {
    int retryCount = 0;
    for (;;) {
      try {
        return objectStore.readObject(storeName, objectId);
      } catch (final ServiceUnavailableException e) {
        if (retryCount++ >= MAX_RETRY_ON_IOERROR) {
          throw e;
        }
      }
      try {
        Thread.sleep(RETRY_WAIT);
      } catch (final InterruptedException e) {
        ;
      }
    }
  }

  /**
   * Open a store and open an object for write. Retry on {@link ServiceUnavailableException}.
   * 
   * @param objectStore
   *          ObjectStoreService reference to use.
   * @param storeName
   *          the name of the store
   * @param objectId
   *          name of object to read.
   * @return a {@link StoreOutputStream} to the object
   * @throws ObjectStoreException
   *           error opening object in the {@link ObjectStoreService}.
   */
  public static StoreOutputStream retryWriteObject(final ObjectStoreService objectStore, final String storeName,
    final String objectId) throws ObjectStoreException {
    int retryCount = 0;
    for (;;) {
      try {
        return objectStore.writeObject(storeName, objectId);
      } catch (final ServiceUnavailableException e) {
        if (retryCount++ >= MAX_RETRY_ON_IOERROR) {
          throw e;
        }
      }
      try {
        Thread.sleep(RETRY_WAIT);
      } catch (final InterruptedException e) {
        ;
      }
    }
  }

  /**
   * Open a store and puts data to an object. Retry on {@link ServiceUnavailableException}.
   * 
   * @param objectStore
   *          ObjectStoreService reference to use.
   * @param storeName
   *          the name of the store
   * @param objectId
   *          name of object to write.
   * @param data
   *          the data to write to the object
   * @throws ObjectStoreException
   *           error opening object in the {@link ObjectStoreService}.
   */
  public static void retryPutObject(final ObjectStoreService objectStore, final String storeName,
    final String objectId, final byte[] data) throws ObjectStoreException {
    int retryCount = 0;
    for (;;) {
      try {
        objectStore.putObject(storeName, objectId, data);
        return;
      } catch (final ServiceUnavailableException e) {
        if (retryCount++ >= MAX_RETRY_ON_IOERROR) {
          throw e;
        }
      }
      try {
        Thread.sleep(RETRY_WAIT);
      } catch (final InterruptedException e) {
        ;
      }
    }
  }

  /**
   * Open a store and gets data to an object. Retry on {@link ServiceUnavailableException}.
   * 
   * @param objectStore
   *          ObjectStoreService reference to use.
   * @param storeName
   *          the name of the store
   * @param objectId
   *          name of object to read.
   * @returns the data of the object
   * @throws ObjectStoreException
   *           error opening object in the {@link ObjectStoreService}.
   */
  public static byte[] retryGetObject(final ObjectStoreService objectStore, final String storeName,
    final String objectId) throws ObjectStoreException {
    int retryCount = 0;
    for (;;) {
      try {
        return objectStore.getObject(storeName, objectId);
      } catch (final ServiceUnavailableException e) {
        if (retryCount++ >= MAX_RETRY_ON_IOERROR) {
          throw e;
        }
      }
      try {
        Thread.sleep(RETRY_WAIT);
      } catch (final InterruptedException e) {
        ;
      }
    }
  }

  /**
   * Prepares a store and retries on {@link ServiceUnavailableException}.
   * 
   * @param objectStore
   *          ObjectStoreService reference to use.
   * @param storeName
   *          the name of the store
   * @param storeProperties
   *          the properties of the store
   * @param storeExistsException
   *          'true': throw exception if store exists
   * @throws ObjectStoreException
   *           error preparing store (even after retrying if it is an {@link ServiceUnavailableException}).
   */
  public static void retryCreateStore(final ObjectStoreService objectStore, final String storeName,
    final AnyMap storeProperties, final boolean storeExistsException) throws ObjectStoreException {
    int retryCount = 0;
    for (;;) {
      try {
        objectStore.createStore(storeName, storeProperties);
        return;
      } catch (final ServiceUnavailableException e) {
        if (retryCount++ >= MAX_RETRY_ON_IOERROR) {
          throw e;
        }
      } catch (final StoreExistsException e) {
        if (storeExistsException) {
          throw e;
        } else {
          return;
        }
      }
      try {
        Thread.sleep(RETRY_WAIT);
      } catch (final InterruptedException e) {
        ;
      }
    }
  }

  /**
   * Ensures that a store exists. If it does not exist, it will be created with default properties.
   * 
   * @param objectStore
   *          ObjectStoreService reference to use.
   * @param storeName
   *          the name of the store
   * @throws ObjectStoreException
   *           error preparing store (even after retrying if it is an {@link ServiceUnavailableException}).
   */
  public static void retryEnsureStore(final ObjectStoreService objectStore, final String storeName)
    throws ObjectStoreException {
    int retryCount = 0;
    for (;;) {
      try {
        objectStore.ensureStore(storeName);
        return;
      } catch (final ServiceUnavailableException e) {
        if (retryCount++ >= MAX_RETRY_ON_IOERROR) {
          throw e;
        }
      }
      try {
        Thread.sleep(RETRY_WAIT);
      } catch (final InterruptedException e) {
        ;
      }
    }
  }

  /**
   * Checks if a store exists.
   * 
   * @param objectStore
   *          ObjectStoreService reference to use.
   * @param storeName
   *          the name of the store
   * @return 'true' if the store exists, 'false' if not.
   * @throws ObjectStoreException
   *           error checking store existence (even after retrying if it is an {@link ServiceUnavailableException}).
   */
  public static boolean retryExistsStore(final ObjectStoreService objectStore, final String storeName)
    throws ObjectStoreException {
    int retryCount = 0;
    for (;;) {
      try {
        return objectStore.existsStore(storeName);
      } catch (final ServiceUnavailableException e) {
        if (retryCount++ >= MAX_RETRY_ON_IOERROR) {
          throw e;
        }
      }
      try {
        Thread.sleep(RETRY_WAIT);
      } catch (final InterruptedException e) {
        ;
      }
    }
  }

  /**
   * Retries listing object infos from a store.
   * 
   * @param objectStore
   *          ObjectStoreService reference to use.
   * @param storeName
   *          the name of the store
   * @param objectIdPrefix
   *          the prefix for the ids of the objects to list.
   * @return the list of object infos for the given prefix and the given store.
   * @throws ObjectStoreException
   *           error reading info (even after retrying if it is an {@link ServiceUnavailableException}).
   */
  public static Collection<StoreObject> retryGetStoreObjectInfos(final ObjectStoreService objectStore,
    final String storeName, final String objectIdPrefix) throws ObjectStoreException {
    int retryCount = 0;
    for (;;) {
      try {
        return objectStore.getStoreObjectInfos(storeName, objectIdPrefix);
      } catch (final ServiceUnavailableException e) {
        if (retryCount++ >= MAX_RETRY_ON_IOERROR) {
          throw e;
        }
      }
      try {
        Thread.sleep(RETRY_WAIT);
      } catch (final InterruptedException e) {
        ;
      }
    }
  }

  /**
   * Checks if an object in a store exists.
   * 
   * @param objectStore
   *          ObjectStoreService reference to use.
   * @param storeName
   *          the name of the store
   * @param objectId
   *          the id of the object
   * @return 'true' if the store exists, 'false' if not.
   * @throws ObjectStoreException
   *           error checking store existence (even after retrying if it is an {@link ServiceUnavailableException}).
   */
  public static boolean retryExistsObject(final ObjectStoreService objectStore, final String storeName,
    final String objectId) throws ObjectStoreException {
    int retryCount = 0;
    for (;;) {
      try {
        return objectStore.existsObject(storeName, objectId);
      } catch (final ServiceUnavailableException e) {
        if (retryCount++ >= MAX_RETRY_ON_IOERROR) {
          throw e;
        }
      }
      try {
        Thread.sleep(RETRY_WAIT);
      } catch (final InterruptedException e) {
        ;
      }
    }
  }

  /**
   * calls StoreOutputStream.close(), retries in case of {@link IOException}.
   * 
   * @throws IOException
   *           could not successfully close {@link StoreOutputStream} even after retrying.
   */
  public static void retryClose(final StoreOutputStream storeOutputStream) throws IOException {
    int retryCount = 0;
    for (;;) {
      try {
        storeOutputStream.close();
        return;
      } catch (final IOException e) {
        if (retryCount++ >= MAX_RETRY_ON_IOERROR) {
          throw e;
        }
        LOG.warn("commit failed, will retry...", e);
      }
      try {
        Thread.sleep(RETRY_WAIT);
      } catch (final InterruptedException e) {
        ;
      }
    }
  }
}
