/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker.input;

import java.io.InputStream;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.CountingInputStream;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;

import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.util.Counters;

/**
 * Provides direct access to the {@link InputStream} for reading from objectstore. Play with each single byte as you
 * like.
 */
public class StreamInput extends Input {

  /** objectstore object stream wrapped in counting stream for later generation of task result counters. */
  private CountingInputStream _stream;

  /** create instance. */
  public StreamInput(final BulkInfo dataObject, final ObjectStoreService objectStore) {
    super(dataObject, objectStore);
  }

  /** get reading stream to objectstore object. */
  public InputStream getStream() throws ObjectStoreException {
    if (_stream == null) {
      final long start = startTime();
      final InputStream objectStream =
        ObjectStoreRetryUtil.retryReadObject(getObjectStore(), getStoreName(), getObjectName());
      _stream = new CountingInputStream(objectStream);
      timeOpen(start);
    }
    return _stream;
  }

  /** close objectstore stream, if open. */
  @Override
  public void close() {
    if (_stream != null) {
      IOUtils.closeQuietly(_stream);
    }
  }

  /** @return size of data read from stream. Is only valid after input is closed. */
  public long getBytesRead() {
    if (_stream != null) {
      return _stream.getByteCount();
    }
    return 0;
  }

  /** @return counter map of superclass extended by the number of bytes read from the stream. */
  @Override
  public Map<String, Number> getCounter() {
    final Map<String, Number> counter = super.getCounter();
    Counters.add(counter, "size", getBytesRead());
    return counter;
  }

  @Override
  public String toString() {
    return "StreamInput: store = " + getStoreName() + ", object = " + getObjectName();
  }

}
