/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.common.definitions;

import java.util.Date;

import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;

/**
 * Base class for job processign element's definitions.
 * 
 */
public class DefinitionBase {

  /** timestamp for this workflow (optional). */
  public static final String KEY_TIMESTAMP = "timestamp";

  /** Property for name in json. */
  public static final String KEY_NAME = "name";

  /** Property for readOnly in json. */
  public static final String KEY_READ_ONLY = "readOnly";

  /** The name of the worker definition (mandatory). */
  protected final String _name;

  /** Any Map from creation, so additional properties from the AnyMap parsed are retained. */
  protected final AnyMap _anyMap;

  /** optional timestamp for this definition. */
  protected final Date _timestamp;

  /** optional read-only flag. */
  protected final boolean _readOnly;

  /**
   * Constructor for WorkerDefinition.
   * 
   * @param definitionAny
   *          The any Object with one definition
   * @throws InvalidDefinitionException
   *           exception if the any object is not filled with all desired values
   */
  public DefinitionBase(final AnyMap definitionAny) throws InvalidDefinitionException {
    _anyMap = definitionAny;
    if (validateName()) {
      _name = AccessAny.getValidName(definitionAny, KEY_NAME);
    } else {
      _name = AccessAny.getStringRequired(definitionAny, KEY_NAME);
    }
    _timestamp = definitionAny.getDateTimeValue(KEY_TIMESTAMP);
    if (definitionAny.containsKey(KEY_READ_ONLY)) {
      _readOnly = definitionAny.getBooleanValue(KEY_READ_ONLY).booleanValue();
    } else {
      _readOnly = false;
    }
  }

  /**
   * Create a new {@link DefinitionBase} from an overriding definition. Timestamp will be set to "now", readonly will be
   * false.
   * 
   * @param name
   *          the name of the definition.
   */
  protected DefinitionBase(final String name) {
    _name = name;
    _timestamp = new Date();
    _readOnly = false;
    _anyMap = null;
  }

  /** override in subclasses to disable name validation. */
  protected boolean validateName() {
    return true;
  }

  /**
   * @return the name
   */
  public String getName() {
    return _name;
  }

  /**
   * @return the timestamp.
   */
  public Date getTimestamp() {
    return _timestamp;
  }

  /**
   * @return the readOnly flag.
   */
  public boolean isReadOnly() {
    return _readOnly;
  }

  /**
   * Returns the object as an Any representation.
   * 
   * @return Any object describing this workflow definition.
   */
  public AnyMap toAny() {
    return toAny(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return toAny().toString();
  }

  /**
   * Returns object information as an Any representation.
   * 
   * @param includingAdditionalAttributes
   *          'true' if also any additional information in the AnyMap should be returned, 'false' if only the (minimal
   *          set of) relevant information should be returned.
   * @return Any object describing this workflow definition.
   */
  public AnyMap toAny(final boolean includingAdditionalAttributes) {
    if (includingAdditionalAttributes && _anyMap != null) {
      return _anyMap;
    } else {
      try {
        final AnyMap baseAny = DataFactory.DEFAULT.createAnyMap();
        baseAny.put(KEY_NAME, _name);
        if (_timestamp != null) {
          baseAny.put(KEY_TIMESTAMP, DataFactory.DEFAULT.createDateTimeValue(_timestamp));
        }
        if (_readOnly) {
          baseAny.put(KEY_READ_ONLY, _readOnly);
        }
        return baseAny;
      } catch (final Exception ex) {
        throw new IllegalStateException("Failed to create Any object for workflow " + _name, ex);
      }
    }
  }
}
