/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.datamodel.ipc;

import java.io.IOException;
import java.io.InputStream;
import java.util.NoSuchElementException;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.Record;

/**
 * Iterator-like class to read a sequence of BON {@link Record} objects from an input stream. Does not implement
 * {@link java.util.Iterator} because the methods throw {@link IOException} on read errors.
 * 
 * @author scum36
 * 
 */
public class BinaryObjectStreamIterator implements AutoCloseable {

  /** the BON stream to read. */
  private final InputStream _binaryStream;

  /** BON parser. */
  private final IpcRecordReader _parser = new IpcRecordReader();

  /** next object to deliver. */
  private Record _nextObject;

  /** goes to true if end-of-stream reached. */
  private boolean _finished;

  /** create instance. */
  public BinaryObjectStreamIterator(final InputStream binaryStream) {
    _binaryStream = binaryStream;
  }

  /**
   * Check if more objects are available on the stream. Exception can only be thrown when called first after the
   * constructor or a call to {@link #next()}, consecutive calls to this method will not throw exception.
   * 
   * @return <code>true</code> if the next call to {@link #next()} will return an {@link Record} object, else
   *         <code>false</code>.
   * 
   * @throws IOException
   *           on stream read errors.
   * @throws IllegalStateException
   *           on BON parse errors.
   */
  public boolean hasNext() throws IOException {
    if (!_finished) {
      try {
        if (_nextObject == null) {
          _nextObject = _parser.readBinaryStream(_binaryStream);
        }
      } finally {
        if (_nextObject == null) {
          close();
        }
      }
    }
    return !_finished;
  }

  /**
   * Get next object from stream. Does not throw any exception if called next after a call to {@link #next()} that
   * returned true.
   * 
   * @return next object from stream
   * @throws NoSuchElementException
   *           no more objects available, a call to {@link #hasNext()} would have returned <code>false</code>.
   * @throws IOException
   *           on stream read errors.
   * @throws IllegalStateException
   *           on BON parse errors.
   */
  public Record next() throws IOException {
    if (_nextObject == null && !hasNext()) {
      throw new NoSuchElementException("No more objects available");
    }

    final Record result = _nextObject;
    _nextObject = null;
    return result;
  }

  /** close the underlying stream. */
  @Override
  public void close() {
    _nextObject = null;
    _finished = true;
    IOUtils.closeQuietly(_binaryStream);
  }
}
