/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker.output;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.CountingOutputStream;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.StoreOutputStream;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.util.Counters;

/**
 * Provides direct access to the {@link OutputStream} of a data object for writing to objectstore.
 */
public class StreamOutput extends Output {

  /** the output stream. */
  private StoreOutputStream _dosStream;

  /** for counting the data size. */
  private CountingOutputStream _stream;

  /**
   * @param dataObject
   *          The data object
   * @param objectStore
   *          The reference to the object store service
   */
  public StreamOutput(final BulkInfo dataObject, final ObjectStoreService objectStore) {
    super(dataObject, objectStore);
  }

  /**
   * @return {@link OutputStream} to the objectstore object.
   */
  public OutputStream getStream() throws ObjectStoreException {
    if (_dosStream == null) {
      final long start = startTime();
      _dosStream = ObjectStoreRetryUtil.retryWriteObject(getObjectStore(), getStoreName(), getObjectName());
      _stream = new CountingOutputStream(_dosStream);
      timeOpen(start);
    }
    return _stream;
  }

  /**
   * Commit the data object to objectstore. Called by the WorkerManager after the task was finished.
   */
  @Override
  public void commit() throws IOException {
    if (_dosStream != null) {
      ObjectStoreRetryUtil.retryClose(_dosStream);
      IOUtils.closeQuietly(_stream);
    }
  }

  /** @return size of data written to the stream. Valid only after committing the input. */
  public long getBytesWritten() {
    if (_stream != null) {
      return _stream.getByteCount();
    }
    return 0;
  }

  /** @return counter map of superclass extended by the number of bytes written to the stream. */
  @Override
  public Map<String, Number> getCounter() {
    final Map<String, Number> counter = super.getCounter();
    Counters.add(counter, "size", getBytesWritten());
    return counter;
  }

  /**
   * {@inheritDoc}
   * 
   * @throws IOException
   */
  @Override
  public void abort() throws IOException {
    if (_dosStream != null) {
      _dosStream.abort();
      ObjectStoreRetryUtil.retryClose(_dosStream);
      IOUtils.closeQuietly(_stream);
    }
  }
}
