/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.impl.base;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.ConnectException;
import java.net.URLEncoder;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.ClientConnectionManager;
import org.apache.http.entity.StringEntity;
import org.apache.http.params.HttpParams;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.client.Attachments;
import org.eclipse.smila.http.client.BulkResponse;
import org.eclipse.smila.http.client.HttpMethod;
import org.eclipse.smila.http.client.RestClient;
import org.eclipse.smila.http.client.RestException;
import org.eclipse.smila.http.client.attachments.RecordAttachments;
import org.eclipse.smila.http.client.util.HttpClientUtil;

/** Default implementation of {@link RestClient}. */
public class RestClientBase implements RestClient {
  /** Connect to a SMILA on the same host on default port. */
  protected static final String DEFAULT_HOSTANDPORT = "http://localhost:8080";

  /** The HttpClient. */
  private final HttpClient _client;

  /** The HttpClient connection manager. */
  private final ClientConnectionManager _connectionManager;

  /** Factory to create Apache HTTP Client related objects. */
  private HttpRequestFactory _requestFactory;

  /** Execution of HTTP requests. */
  private HttpRequestExecutor _requestExecutor;

  /** Handling of HTTP results. */
  private HttpResultHandler _resultHandler;

  /** The logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** create instance with the given sub-objects. */
  public RestClientBase(final ClientConnectionManager connectionManager, final HttpRequestFactory requestFactory,
    final HttpRequestExecutor requestExecutor, final HttpResultHandler resultHandler) {
    _connectionManager = connectionManager;
    _requestFactory = requestFactory;
    _requestExecutor = requestExecutor;
    _resultHandler = resultHandler;
    _client = HttpClientUtil.createInstance(_connectionManager);
  }

  /**
   * reset the {@link HttpRequestFactory} for this client. Not needed for standard use cases but useful for custom
   * extensions of the client.
   */
  public void setRequestFactory(final HttpRequestFactory requestFactory) {
    _requestFactory = requestFactory;
  }

  /**
   * reset the {@link HttpRequestExecutor} for this client. Not needed for standard use cases but useful for custom
   * extensions of the client.
   */
  public void setRequestExecutor(final HttpRequestExecutor requestExecutor) {
    _requestExecutor = requestExecutor;
  }

  /**
   * reset the {@link HttpResultHandler} for this client. Not needed for standard use cases but useful for custom
   * extensions of the client.
   */
  public void setResultHandler(final HttpResultHandler resultHandler) {
    _resultHandler = resultHandler;
  }

  @Override
  public void setClientParameter(final String name, final Object value) {
    _client.getParams().setParameter(name, value);
  }

  @Override
  public void shutdown() {
    _connectionManager.shutdown();
  }

  @Override
  public String getHostAndPort() {
    return _requestExecutor.getHostAndPort();
  }

  @Override
  public AnyMap get(final String resource) throws RestException, IOException {
    return invokeAsMap(HttpMethod.GET, resource, null, null, null);
  }

  @Override
  public AnyMap get(final String resource, final AnyMap parameters) throws RestException, IOException {
    return invokeAsMap(HttpMethod.GET, buildPathAndQuery(resource, parameters), null, null, null);
  }

  @Override
  public BulkResponse getBulk(final String resource, final HttpParams methodParams) throws IOException,
    RestException {
    final String url = createUrl(resource);
    final HttpUriRequest request = _requestFactory.getHttpMethod(HttpMethod.GET, url, methodParams);
    try {
      final HttpResponse response = _requestExecutor.execute(_client, request);
      final InputStream content = _resultHandler.processHttpResponse(request, response);
      return _resultHandler.handleJsonBulkResult(content);
    } catch (final ConnectException e) {
      request.abort();
      throw new ConnectException(createConnectErrorMessage(url, e));
    }
  }

  @Override
  public AnyMap post(final String resource) throws RestException, IOException {
    return post(resource, null, null);
  }

  @Override
  public AnyMap post(final String resource, final AnyMap parameters) throws RestException, IOException {
    return post(resource, parameters, null);
  }

  @Override
  public AnyMap post(final String resource, final AnyMap parameters, final Attachments attachments)
    throws RestException, IOException {
    return invokeAsMap(HttpMethod.POST, resource, parameters, attachments, null);
  }

  @Override
  public AnyMap post(final String resource, final Record record) throws RestException, IOException {
    Attachments attachments = null;
    if (record.hasAttachments()) {
      attachments = new RecordAttachments(record);
    }
    return post(resource, record.getMetadata(), attachments);
  }

  @Override
  public AnyMap post(final String resource, final String parameters, final String encoding, final String contentType)
    throws RestException, IOException {
    return asMap(doRequest(HttpMethod.POST, resource, new StringEntity(parameters, contentType, encoding), null));
  }

  @Override
  public AnyMap put(final String resource, final AnyMap parameters) throws RestException, IOException {
    return invokeAsMap(HttpMethod.PUT, resource, parameters, null, null);
  }

  @Override
  public AnyMap put(final String resource, final Record record) throws RestException, IOException {
    return put(resource, record.getMetadata());
  }

  @Override
  public AnyMap delete(final String resource) throws RestException, IOException {
    return invokeAsMap(HttpMethod.DELETE, resource, null, null, null);
  }

  @Override
  public AnyMap delete(final String resource, final AnyMap parameters) throws RestException, IOException {
    return invokeAsMap(HttpMethod.DELETE, buildPathAndQuery(resource, parameters), null, null, null);
  }

  @Override
  public Any invoke(final HttpMethod method, final String resource, final InputStream inputStream,
    final HttpParams params) throws RestException, IOException {
    final HttpEntity entity = _requestFactory.createJsonEntity(inputStream);
    return doRequest(method, resource, entity, params);
  }

  @Override
  public Any invoke(final HttpMethod method, final String resource, final InputStream inputStream,
    final String contentType, final HttpParams params) throws RestException, IOException {
    final HttpEntity entity = _requestFactory.createEntity(inputStream, contentType);
    return doRequest(method, resource, entity, params);
  }

  @Override
  public Any invoke(final HttpMethod method, final String resource, final AnyMap parameters,
    final Attachments attachments, final HttpParams httpParams) throws RestException, IOException {
    final HttpEntity requestEntity = _requestFactory.createHttpEntity(parameters, attachments);
    return doRequest(method, resource, requestEntity, null);
  }

  @Override
  public Any invoke(final HttpMethod method, final String resource, final AnySeq parameters,
    final Attachments attachments, final HttpParams httpParams) throws RestException, IOException {
    final HttpEntity requestEntity = _requestFactory.createHttpEntity(parameters, attachments);
    return doRequest(method, resource, requestEntity, null);
  }

  /**
   * Invoke the named resource and ensure that the result is returned as a map object. If the original was not a map,
   * wrap it in a new map under key "result".
   */
  protected AnyMap invokeAsMap(final HttpMethod method, final String resource, final AnyMap parameters,
    final Attachments attachments, final HttpParams httpParams) throws RestException, IOException {
    return asMap(invoke(method, resource, parameters, attachments, httpParams));
  }

  /**
   * Ensure that the result is returned as a map object. If the original was not a map, wrap it in a new map under key
   * "result".
   */
  protected AnyMap asMap(final Any result) throws RestException, IOException {
    if (result == null) {
      return null;
    }
    if (result.isMap()) {
      return result.asMap();
    }
    final AnyMap resultAsMap = result.getFactory().createAnyMap();
    resultAsMap.put("result", result);
    return resultAsMap;
  }

  /** Execute a request, try to parse a JSON result, create appropriate exceptions. */
  protected Any doRequest(final HttpMethod method, final String resource, final HttpEntity requestEntity,
    final HttpParams params) throws RestException, IOException {

    final String url = createUrl(resource);
    final HttpUriRequest request = _requestFactory.getHttpMethod(method, url, params);

    try {
      if (requestEntity != null) {
        if (request instanceof HttpEntityEnclosingRequest) {
          ((HttpEntityEnclosingRequest) request).setEntity(requestEntity);
        } else {
          _log.warn("Method " + method + " does not support sending a request body, input is ignored.");
        }
      }
      final HttpResponse response = _requestExecutor.execute(_client, request);
      final InputStream content = _resultHandler.processHttpResponse(request, response);
      return _resultHandler.handleJsonResult(content);
    } catch (final ConnectException e) {
      request.abort();
      throw new ConnectException(createConnectErrorMessage(url, e));
    }
  }

  /** Create url with correct slashes. */
  protected String createUrl(final String resource) {
    if (resource.startsWith("/")) {
      return _requestExecutor.getHostAndPort() + resource;
    } else {
      return _requestExecutor.getHostAndPort() + "/" + resource;
    }
  }

  /** Create an error message for connection errors including the invoked URL. */
  protected String createConnectErrorMessage(final String url, final ConnectException e) {
    String msg = e.getMessage();
    if (msg == null) {
      msg = "";
    }
    msg += " on " + url;
    return msg;
  }

  protected String buildPathAndQuery(final String resource, final AnyMap parameters) {
    if (parameters.isEmpty()) {
      return resource;
    }
    final StringBuilder query = new StringBuilder("?");
    for (final Map.Entry<String, Any> entry : parameters.entrySet()) {
      if (entry.getValue().isValue()) {
        try {
          final String name = URLEncoder.encode(entry.getKey(), "UTF-8");
          final String value = URLEncoder.encode(entry.getValue().asValue().asString(), "UTF-8");
          if (query.length() > 1) {
            query.append("&");
          }
          query.append(name).append("=").append(value);
        } catch (final UnsupportedEncodingException ex) {
          // this. is. not. possible.
        }
      }
    }
    return resource + query.toString();
  }
}
