/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky, Andreas Schank (both Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.util;

import org.apache.http.client.HttpClient;
import org.apache.http.conn.ClientConnectionManager;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.conn.tsccm.ThreadSafeClientConnManager;

/**
 * Utility class for HttpClient instance creation. Use this class instead of direct HttpClient constructor calls.
 * 
 * @since 1.1.0
 */
public final class HttpClientUtil {
  /** Private default Constructor to avoid instance creation. */
  private HttpClientUtil() {
  }

  /** Create a new HttpClient instance with default settings and HttpMethodRetry disabled. */
  public static HttpClient createInstance() {

    final DefaultHttpClient client = new DefaultHttpClient();
    disableRetries(client);
    return client;
  }

  /**
   * Create a new HttpClient instance with default settings using the given ClientConnectionManager and with
   * HttpMethodRetry disabled.
   * 
   * @param connectionManager
   *          the ClientConnectionManager to use
   * @return a new HttpClient instance
   */
  public static HttpClient createInstance(final ClientConnectionManager connectionManager) {
    final DefaultHttpClient client = new DefaultHttpClient(connectionManager);
    disableRetries(client);
    return client;
  }

  /**
   * Disable HttpMethodRetry on the given DefaultHttpClient.
   * 
   * @param client
   *          an HttpClient instance
   */
  private static void disableRetries(final DefaultHttpClient client) {
    final DefaultHttpRequestRetryHandler retryhandler = new DefaultHttpRequestRetryHandler(0, false);
    client.setHttpRequestRetryHandler(retryhandler);
  }

  /** Create a default connection manager usable for multithreading. */
  public static ClientConnectionManager createThreadSafeConnectionManager() {
    return new ThreadSafeClientConnManager();
  }

  /** Create a default connection manager usable for multithreading with given connection limits. */
  public static ClientConnectionManager createThreadSafeConnectionManager(final int maxTotalConnections,
    final int maxConnectionsPerHost) {
    final ThreadSafeClientConnManager manager = new ThreadSafeClientConnManager();
    manager.setMaxTotal(maxTotalConnections);
    manager.setDefaultMaxPerRoute(maxConnectionsPerHost);
    return manager;
  }

}
