/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file.internal;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.DirectoryStream.Filter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.ImportingException;
import org.eclipse.smila.importing.crawler.file.FileCrawlerService;
import org.eclipse.smila.importing.util.PropertyNameMapper;
import org.eclipse.smila.taskworker.TaskContext;

/**
 * Default implementation of a FileCrawlerService.
 * 
 * @author stuc07
 * 
 */
public class FileCrawlerServiceImpl implements FileCrawlerService {

  private final Set<String> _filePropertyNames = new HashSet<String>();

  public FileCrawlerServiceImpl() {
    _filePropertyNames.add(PROPERTY_FILE_NAME);
    _filePropertyNames.add(PROPERTY_FILE_PATH);
    _filePropertyNames.add(PROPERTY_FILE_FOLDER);
    _filePropertyNames.add(PROPERTY_FILE_SIZE);
    _filePropertyNames.add(PROPERTY_FILE_LAST_MODIFIED);
    _filePropertyNames.add(PROPERTY_FILE_EXTENSION);
    _filePropertyNames.add(ATTACHMENT_FILE_CONTENT);
  }

  private static class FileFilter implements Filter<Path> {

    @Override
    public boolean accept(final Path entry) throws IOException {
      return Files.isRegularFile(entry);
    }

  }

  private static class DirectoryFilter implements Filter<Path> {

    @Override
    public boolean accept(final Path entry) throws IOException {
      return Files.isDirectory(entry);
    }

  }

  @Override
  public Collection<Path> listFiles(final Path directory) throws IOException {
    final Collection<Path> result = new ArrayList<>();
    try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(directory, new FileFilter())) {
      final Iterator<Path> iter = directoryStream.iterator();
      while (iter.hasNext()) {
        result.add(iter.next());
      }
    }
    return result;
  }

  @Override
  public Collection<Path> listDirectories(final Path directory) throws IOException {
    final Collection<Path> result = new ArrayList<>();
    try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(directory, new DirectoryFilter())) {
      final Iterator<Path> iter = directoryStream.iterator();
      while (iter.hasNext()) {
        result.add(iter.next());
      }
    }
    return result;
  }

  @Override
  public Collection<Path> list(final Path directory) throws IOException {
    final Collection<Path> result = new ArrayList<>();
    try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(directory)) {
      final Iterator<Path> iter = directoryStream.iterator();
      while (iter.hasNext()) {
        result.add(iter.next());
      }
    }
    return result;
  }

  @Override
  public Record fileToRecord(final Path file, final String dataSource, final boolean withContent)
    throws IOException {
    final FileToRecordConverter converter = new FileToRecordConverter(DataFactory.DEFAULT);
    return converter.fileToRecord(file, dataSource, withContent);
  }

  @Override
  public void addAttachment(final Record record, final String pathAttribute, final String attachmentName)
    throws IOException {
    final FileToRecordConverter converter = new FileToRecordConverter(DataFactory.DEFAULT);
    converter.addAttachment(record, pathAttribute, attachmentName);
  }

  @Override
  public Record dirToRecord(final Path directory, final String dataSource) throws IOException {
    final FileToRecordConverter converter = new FileToRecordConverter(DataFactory.DEFAULT);
    return converter.dirToRecord(directory, dataSource);
  }

  @Override
  public InputStream getContent(final Record record, final TaskContext taskContext) throws ImportingException {
    final PropertyNameMapper mapper = PropertyNameMapper.createFrom(taskContext);
    final String path =
      record.getMetadata().getStringValue(mapper.get(FileCrawlerService.PROPERTY_FILE_PATH).get(0));
    if (path == null) {
      throw new IllegalArgumentException("Record '" + record.getId() + "' does not contain attribute '"
        + mapper.get(FileCrawlerService.PROPERTY_FILE_PATH).get(0) + "'");
    }
    final Path file = Paths.get(path);
    if (!Files.isRegularFile(file)) {
      throw new ImportingException("File '" + file + "' is not a file with content.");
    }
    try {
      return Files.newInputStream(file, StandardOpenOption.READ);
    } catch (final IOException ex) {
      throw new ImportingException("File '" + file + "' does not exist or is not readable.", ex, false);
    }
  }

  @Override
  public Set<String> getFilePropertyNames() {
    return _filePropertyNames;
  }

}
