/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager.httphandler;

import java.util.List;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;

import org.eclipse.smila.taskmanager.BadParameterTaskmanagerException;
import org.eclipse.smila.taskmanager.TaskManager;

/**
 * Implements the handling of HTTP requests for the task state API.
 * 
 * URL pattern: /smila/tasks/--workername--/inprogress/--taskid-- or /smila/tasks/--workername--/todo/--taskid--
 */
public class TaskPathStateHandler extends JsonRequestHandler {

  /** The reference to the TaskManager service. */
  private TaskManager _taskManager;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    if ("GET".equals(method)) {
      final String workerName = getWorkerName(requestUri);
      final String section = getSection(requestUri);
      final String taskName = getTaskName(requestUri);
      return _taskManager.getTaskInfo(workerName, section, taskName);
    }
    return null;
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for IllegalArgumentException to the exception handling of
   * {@link JsonRequestHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof IllegalArgumentException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex.getCause() != null && ex.getCause() instanceof BadParameterTaskmanagerException) {
      return HttpStatus.NOT_FOUND;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * Gets the worker name from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return the worker name from the requestUri.
   */
  private String getWorkerName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * Gets the task queue name from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return the task queue name from the requestUri.
   */
  private String getSection(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(1);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * Gets the task id from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return the task id name from the requestUri.
   */
  private String getTaskName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    final StringBuilder taskName = new StringBuilder();
    if (uriParts.size() > 2 && uriParts.get(2) != null) {
      taskName.append(uriParts.get(2));
    }
    final int uriSizeForQualifiedTasks = 3;
    // check if there are more parts (e.g. qualifier)
    if (uriParts.size() > uriSizeForQualifiedTasks && uriParts.get(uriSizeForQualifiedTasks) != null) {
      // e.g. requestUri = "/smila/tasks/hssiRecorddeleter/todo_qualified/parts_--UUID--/--task_id--/"
      taskName.append(uriParts.get(uriSizeForQualifiedTasks));
    }
    if (taskName.length() > 0) {
      return taskName.toString();
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * {@inheritDoc}
   * 
   * GET and DELETE are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return ("GET".equals(method));
  }

  /**
   * @param taskManager
   *          the reference to the JobManager service.
   */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /**
   * @param taskManager
   *          the reference to the current JobManager service.
   */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (_taskManager == taskManager) {
      _taskManager = null;
    }
  }
}
