/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation
 *               Andreas Weber (Attensity Europe GmbH) - data model simplification
 **********************************************************************************************************************/

package org.eclipse.smila.processing.pipelets.xmlprocessing;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.pipelets.xmlprocessing.util.XslTransformer;
import org.w3c.dom.Document;

/**
 * Pipelet that performs an XSL transformation on an attribute or attachment value. The possible properties are:
 * <ul>
 * <li>xslFile: the name of the XSLT file</li>
 * <li>inputName: name of the Attribute/Attachment to apply the XSL transformation to</li>
 * <li>outputName: name of the Attribute/Attachment to store the XSL transformation in</li>
 * <li>inputType: the type (Attribute or Attachment of the inputName</li>
 * <li>outputType: the type (Attribute or Attachment of the outputtName</li>
 * </ul>
 */
public class XslTransformationPipelet extends AXmlTransformationPipelet {
  /**
   * The name of the XSLT file used for the transformation.
   */
  public static final String PROP_XSL_FILE = "xslFile";

  /**
   * The name of the property that indicates to add the attributes of the current record as XSLT parameters.
   */
  public static final String PROP_PARAMETERS = "parameters";

  /**
   * XSLT transformer.
   */
  private XslTransformer _xslTransformer = new XslTransformer();

  @Override
  protected void processRecord(final Blackboard blackboard, final ParameterAccessor paramAccessor, final String id)
    throws Exception {

    final String xslFile = paramAccessor.getRequiredParameter(PROP_XSL_FILE);

    final Document inputDocument = createDocument(blackboard, id, paramAccessor);
    if (inputDocument != null) {
      Any parameters = paramAccessor.getParameterAny(PROP_PARAMETERS);
      if (parameters != null) {
        if (parameters.isBoolean()) {
          if (parameters.asValue().asBoolean()) {
            parameters = blackboard.getMetadata(id);
          } else {
            parameters = null;
          }
        }
      }
      final Document result =
        _xslTransformer.transform(inputDocument, xslFile, parameters != null ? parameters.asMap() : null);
      storeDocument(blackboard, id, result, paramAccessor);
    }
  }
}
