/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker.internal;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.TaskLog;
import org.eclipse.smila.taskworker.input.Inputs;
import org.eclipse.smila.taskworker.output.OutputMode;
import org.eclipse.smila.taskworker.output.Outputs;
import org.eclipse.smila.taskworker.util.Counters;

/**
 * Implementation of TaskContext.
 */
public class TaskContextImpl implements TaskContext {

  /** The task used in the context. */
  private final Task _task;

  /** The inputs. */
  private final Inputs _inputs;

  /** The outputs. */
  private final Outputs _outputs;

  /** worker specific durations. */
  private final Map<String, Number> _workerDurations = new HashMap<String, Number>();

  /** worker specific counters. */
  private final Map<String, Number> _workerCounters = new HashMap<String, Number>();

  /** True if canceled, false else. */
  private boolean _canceled;

  /** task logger. */
  private final TaskLog _taskLog;

  /** The reference to the object store service. */
  private final ObjectStoreService _objectStore;

  /** create instance. */
  public TaskContextImpl(final Task task, final TaskLog taskLog, final ObjectStoreService objectStore) {
    this(task, taskLog, objectStore, new HashMap<String, Collection<OutputMode>>(0));
  }

  /** create instance. */
  public TaskContextImpl(final Task task, final TaskLog taskLog, final ObjectStoreService objectStore,
    final Map<String, Collection<OutputMode>> outputModes) {
    _task = task;
    _taskLog = taskLog;
    _objectStore = objectStore;
    _inputs = new Inputs(task.getInputBulks(), objectStore);
    _outputs = new Outputs(task.getOutputBulks(), objectStore);
    _outputs.setOutputModes(outputModes);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Task getTask() {
    return _task;
  }

  @Override
  public AnyMap getTaskParameters() {
    return _task.getParameters();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Inputs getInputs() {
    return _inputs;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Outputs getOutputs() {
    return _outputs;
  }

  @Override
  public Map<String, Number> getFinalCounters() {
    final Map<String, Number> counters = new HashMap<String, Number>();
    Counters.add(counters, Counters.WARN_COUNT, _taskLog.getWarnCount());
    _inputs.addInputCounters(counters);
    _outputs.addOutputCounters(counters);
    if (counters.containsKey(Counters.DURATION_IODATA_CLOSE)) {
      Counters.add(counters, Counters.DURATION_IODATA, counters.get(Counters.DURATION_IODATA_CLOSE).doubleValue());
    }
    if (counters.containsKey(Counters.DURATION_IODATA_OPEN)) {
      Counters.add(counters, Counters.DURATION_IODATA, counters.get(Counters.DURATION_IODATA_OPEN).doubleValue());
    }
    if (!_workerDurations.isEmpty()) {
      Counters.addAll(counters, _workerDurations, Counters.DURATION_PERFORM_FUNCTION);
    }
    if (!_workerCounters.isEmpty()) {
      Counters.addAll(counters, _workerCounters, Counters.FUNCTION);
    }
    return counters;
  }

  @Override
  public void addToCounter(final String name, final long value) {
    Counters.add(_workerCounters, name, value);
  }

  @Override
  public void addToCounter(final String name, final double value) {
    Counters.add(_workerCounters, name, value);
  }

  @Override
  public long getTimestamp() {
    return System.nanoTime();
  }

  @Override
  public void measureTime(final String name, final long startTime) {
    final long duration = System.nanoTime() - startTime;
    addDuration(name, duration);
  }

  @Override
  public void addDuration(final String name, final long duration) {
    Counters.addDuration(_workerDurations, name, duration);
  }

  @Override
  public void addDuration(final String name, final double duration) {
    Counters.add(_workerDurations, name, duration);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isCanceled() {
    return _canceled;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void cancel() {
    _canceled = true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ObjectStoreService getObjectStore() {
    return _objectStore;
  }

  @Override
  public TaskLog getLog() {
    return _taskLog;
  }

}
