/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.solr.server;

import java.util.HashMap;
import java.util.Map;

import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;

/** manages one {@link SolrServer} instance per core. */
public abstract class SolrServers {
  /** map which hold all running servers. */
  private final Map<String, SolrServer> _servers = new HashMap<String, SolrServer>();

  /**
   * create a SolrServer for admin operations not related to a single core (e.g. getting core names). This server is not
   * cached.
   */
  public abstract SolrServer getAdminServer() throws SolrServerException;

  /** create (if necessary), cache and return a {@link SolrServer} for the given core. */
  public SolrServer getSolrServer(String coreName) throws SolrServerException {
    SolrServer server = _servers.get(coreName);
    if (server == null) {
      synchronized (_servers) {
        server = createServer(coreName);
        _servers.put(coreName, server);
      }
    }
    return server;
  }

  /** remove a {@link SolrServer} that is not needed anymore. */
  public void removeSolrServer(String coreName) {
    synchronized (_servers) {
      _servers.remove(coreName);
    }
  }

  /** remove all {@link SolrServer}s. */
  public void removeAllServers() {
    synchronized (_servers) {
      _servers.clear();
    }
  }

  /** create correct {@link SolrServer} instance for the given core. */
  protected abstract SolrServer createServer(String coreName) throws SolrServerException;
}
