/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.processing.pipelets;

import java.io.ByteArrayOutputStream;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;

/**
 * Writes an {@link Any} object to JSON.
 * 
 * Copyright (c) 2012 Attensity Europe GmbH
 * 
 * @author Tobias Liefke
 */
public class JSONWriterPipelet implements Pipelet {

  /** Name of the property that contains the names of the input attributes. */
  public static final String INPUT_ATTRIBUTES_PROPERTY = "inputAttributes";

  /** Name of the property that contains the type of the output (attribute/attachment). */
  public static final String OUTPUT_NAME_PROPERTY = ATransformationPipelet.PROP_OUTPUT_NAME;

  /** Name of the property that contains the name of the output attribute or attachment. */
  public static final String OUTPUT_TYPE_PROPERTY = ATransformationPipelet.PROP_OUTPUT_TYPE;

  /** Name of the property that indicates to format the output for better readability. */
  public static final String PRINT_PRETTY_PROPERTY = "printPretty";

  /** The list of input attributes to read. */
  private List<String> _inputAttributes;

  /** Indicates that the content of the given attribute is used, instead of a new map with the given attributes. */
  private boolean _plain;

  /** The name of the result attribute/attachment. */
  private String _outputName;

  /** The type of the result. */
  private SourceType _outputType;

  /** Indicates to format the output for better readability. */
  private boolean _printPretty;

  /** Our configuration. */
  private AnyMap _config;

  /** Our logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /**
   * Initializes parameters.
   * 
   * @see Pipelet#configure(AnyMap)
   */
  @Override
  public void configure(AnyMap config) throws ProcessingException {
    this._config = config;
    final ParameterAccessor paramAccessor = new ParameterAccessor(config);

    _inputAttributes = paramAccessor.getParameters(INPUT_ATTRIBUTES_PROPERTY);
    _plain = !(config.get(INPUT_ATTRIBUTES_PROPERTY) instanceof AnySeq);

    _outputName = paramAccessor.getRequiredParameter(OUTPUT_NAME_PROPERTY);
    final String outputType = paramAccessor.getRequiredParameter(OUTPUT_TYPE_PROPERTY);
    try {
      this._outputType = SourceType.valueOf(outputType);
    } catch (IllegalArgumentException e) {
      throw new ProcessingException("Illegal value for output type: " + outputType);
    }

    _printPretty = paramAccessor.getBooleanParameter(PRINT_PRETTY_PROPERTY, Boolean.TRUE);
  }

  /**
   * @see Pipelet#process(Blackboard, String[])
   */
  @Override
  public String[] process(Blackboard blackboard, String[] recordIds) throws ProcessingException {
    final ResultCollector results =
      new ResultCollector(new ParameterAccessor(_config), _log, ProcessingConstants.DROP_ON_ERROR_DEFAULT);
    final IpcAnyWriter writer = new IpcAnyWriter(_printPretty);
    for (final String id : recordIds) {
      try {
        // Determine input
        final AnyMap metadata = blackboard.getMetadata(id);
        Any object;
        if (_plain) {
          if (_inputAttributes.size() == 0) {
            object = metadata;
          } else {
            object = metadata.get(_inputAttributes.get(0));
          }
        } else {
          final AnyMap map = blackboard.getDataFactory().createAnyMap();
          for (String attribute : _inputAttributes) {
            if (metadata.containsKey(attribute)) {
              map.put(attribute, metadata.get(attribute));
            }
          }
          object = map;
        }

        // And write to output attribute / attachment
        if (_outputType == SourceType.ATTACHMENT) {
          final ByteArrayOutputStream buffer = new ByteArrayOutputStream();
          writer.writeJsonStream(object, buffer);
          blackboard.setAttachment(id, _outputName, buffer.toByteArray());
        } else {
          metadata.put(_outputName, writer.writeJsonObject(object));
        }

        results.addResult(id);
      } catch (Exception e) {
        results.addFailedResult(id, e);
      }
    }
    return results.getResultIds();
  }

}
