/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file.filter;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.importing.util.RegexPatternMatcher;

/** container class for accessing filter configuration. */
public class FilterConfiguration {

  public static final String MAX_SIZE = "maxFileSize";

  public static final String MAX_DEPTH = "maxFolderDepth";

  public static final String SYMBOLIC_LINKS = "followSymbolicLinks";

  public static final String FILE_PATTERNS = "filePatterns";

  public static final String FOLDER_PATTERNS = "folderPatterns";

  public static final String INCLUDE_PATTERNS = "include";

  public static final String EXCLUDE_PATTERNS = "exclude";

  /** matcher for checking include and exclude patterns of files. */
  private final RegexPatternMatcher _fileMatcher = new RegexPatternMatcher();

  /** matcher for checking include and exclude patterns of folders. */
  private final RegexPatternMatcher _folderMatcher = new RegexPatternMatcher();

  /** max size. */
  private long _maxSize = Long.MAX_VALUE;

  /** max depth. */
  private long _maxDepth = Long.MAX_VALUE;

  /** whether to follow symbolic links. */
  private boolean _symbolicLinks;

  /**
   * @param filterConfig
   *          filter section from file crawler configuration.
   */
  public FilterConfiguration(final AnyMap filterConfig) {
    if (filterConfig.containsKey(MAX_SIZE)) {
      _maxSize = filterConfig.getLongValue(MAX_SIZE);
    }
    if (filterConfig.containsKey(MAX_DEPTH)) {
      _maxDepth = filterConfig.getLongValue(MAX_DEPTH);
    }
    if (filterConfig.containsKey(SYMBOLIC_LINKS)) {
      _symbolicLinks = filterConfig.getBooleanValue(SYMBOLIC_LINKS);
    }
    initPatternMatcher(filterConfig, FILE_PATTERNS, _fileMatcher);
    initPatternMatcher(filterConfig, FOLDER_PATTERNS, _folderMatcher);
  }

  /** initialize a pattern matcher from the filter configuration, if configured. */
  private void initPatternMatcher(final AnyMap filterConfig, final String patternKey,
    final RegexPatternMatcher patternMatcher) {
    if (filterConfig.containsKey(patternKey)) {
      final AnyMap patterns = filterConfig.getMap(patternKey);
      if (patterns.containsKey(INCLUDE_PATTERNS)) {
        final Any includes = patterns.get(INCLUDE_PATTERNS);
        for (final Any include : includes) {
          if (include.isValue()) {
            patternMatcher.addIncludePattern(include.asValue().asString());
          }
        }
      }
      if (patterns.containsKey(EXCLUDE_PATTERNS)) {
        final Any excludes = patterns.get(EXCLUDE_PATTERNS);
        for (final Any exclude : excludes) {
          if (exclude.isValue()) {
            patternMatcher.addExcludePattern(exclude.asValue().asString());
          }
        }
      }
    }

  }

  /** @return max size in bytes. */
  public long getMaxSize() {
    return _maxSize;
  }

  /** @return max depth to crawl into. */
  public long getMaxDepth() {
    return _maxDepth;
  }

  /** @return whether to follow symbolic links, default: false. */
  public boolean followSymbolicLinks() {
    return _symbolicLinks;
  }

  /** @return matcher for checking include and exclude patterns of files. */
  public RegexPatternMatcher getFilePatternMatcher() {
    return _fileMatcher;
  }

  /** @return matcher for checking include and exclude patterns of folders. */
  public RegexPatternMatcher getFolderPatternMatcher() {
    return _folderMatcher;
  }

}
