/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.zookeeper.internal;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;
import org.eclipse.smila.clusterconfig.ClusterConfigException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcRecordReader;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.zookeeper.ZkConnection;
import org.eclipse.smila.zookeeper.ZooKeeperService;

/**
 * Http admin handler for Zookeeper.
 */
public class ZooKeeperAdminHandler extends JsonRequestHandler {

  /** key to set a watch when getting a record. */
  public static final String KEY_SET_WATCH = "setWatch";

  /** key of content when PUTting a record. */
  public static final String KEY_DATA = "data";

  /** ZooKeeperService. */
  private ZooKeeperService _zkService;

  /** ZkConnection. */
  private ZkConnection _zkConnection;

  /** for BON -> data serialization. */
  private final IpcRecordReader _ipcRecordReader = new IpcRecordReader();

  /** for data -> BON serialization. */
  private final IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /**
   * Default Constructor.
   */
  public ZooKeeperAdminHandler() {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "PUT".equals(method) || "DELETE".equals(method);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord,
    final HttpExchange exchange) throws Exception {
    final String path = getPath(requestUri);
    final Record result = FACTORY.createRecord();
    final ZooKeeper client = _zkService.getClient();
    if ("GET".equals(method)) {
      final boolean setWatch;
      if (inputRecord != null && inputRecord.getMetadata().containsKey(KEY_SET_WATCH)) {
        setWatch = inputRecord.getMetadata().getBooleanValue(KEY_SET_WATCH).booleanValue();
      } else {
        setWatch = false;
      }
      final Stat stat = client.exists(path, setWatch);
      if (stat != null) {
        result.getMetadata().put("stat", convertStat(stat));
        if (stat.getDataLength() > 0) {
          result.getMetadata().put("data", getData(client, path));
        }
        if (stat.getNumChildren() > 0) {
          result.getMetadata().put("children", getChildren(client, path, requestUri, getRequestHost(exchange)));
        }
      }
    } else if ("PUT".equals(method)) {
      final byte[] data;
      if (inputRecord != null) {
        data = _ipcRecordWriter.writeBinaryObject(inputRecord);
      } else {
        data = new byte[0];
      }
      try {
        _zkConnection.createPath(path, data);
      } catch (final KeeperException.NodeExistsException e) {
        _zkConnection.setData(path, data);
      }
    } else if ("DELETE".equals(method)) {
      if (_log.isInfoEnabled()) {
        _log.info("Deleting zookeeper path: " + path);
      }
      _zkConnection.deleteTree(path);
    }
    return result;
  }

  /**
   * @return zookeeper node data converted to record metadata
   */
  private Any getData(final ZooKeeper client, final String path) {
    final byte[] data;
    try {
      data = client.getData(path, false, null);
    } catch (final Exception ex) {
      return FACTORY.createStringValue("Failed to read data: " + ex.toString());
    }
    try {
      final Record record = _ipcRecordReader.readBinaryObject(data);
      return record.getMetadata();
    } catch (final Exception bonEx) {
      try {
        final Record record = _ipcRecordReader.readJsonStream(new ByteArrayInputStream(data));
        return record.getMetadata();
      } catch (final Exception jsonEx) {
        try {
          return FACTORY.createStringValue(new String(data, "utf-8"));
        } catch (final Exception stringEx) {
          return FACTORY.createStringValue("Data not in BON or JSON format, nor a UTF-8 string.");
        }
      }
    }
  }

  /**
   * @return children of a zookeeper node converted to AnySeq.
   */
  private AnySeq getChildren(final ZooKeeper client, final String path, final String requestUri,
    final String requestHost) throws KeeperException, InterruptedException, MalformedURLException {
    final List<String> children = client.getChildren(path, false);
    final AnySeq result = FACTORY.createAnySeq();
    for (int i = 0; i < children.size(); ++i) {
      final URL url = URLCreator.create(requestHost, requestUri, children.get(i));
      result.add(url.toString());
    }
    return result;
  }

  /**
   * @return stat data of a zookeeper node converted to AnyMap.
   */
  private AnyMap convertStat(final Stat stat) {
    final AnyMap map = FACTORY.createAnyMap();
    map.put("a-version", stat.getAversion());
    map.put("c-time", FACTORY.createDateTimeValue(new Date(stat.getCtime())));
    map.put("c-version", stat.getCversion());
    map.put("c-zxid", stat.getCzxid());
    map.put("data-length", stat.getDataLength());
    map.put("ephemeral-owner", stat.getEphemeralOwner());
    map.put("m-time", FACTORY.createDateTimeValue(new Date(stat.getMtime())));
    map.put("m-zxid", stat.getMzxid());
    map.put("num-children", stat.getNumChildren());
    map.put("p-zxid", stat.getPzxid());
    map.put("version", stat.getVersion());
    return map;
  }

  /**
   * @return zookeeper path for given request
   */
  protected String getPath(final String requestUri) {
    String path = "/";
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      path = uriParts.get(0);
      if (!path.startsWith("/")) {
        path = "/" + path;
      }
      if (path.length() > 1 && path.endsWith("/")) {
        path = path.substring(0, path.length() - 1);
      }
    }
    return path;
  }

  /**
   * 
   * @param zooKeeperService
   *          referenced service
   */
  public void setZooKeeperService(final ZooKeeperService zooKeeperService) {
    _zkService = zooKeeperService;
    _zkConnection = new ZkConnection(_zkService);
  }

  /**
   * 
   * @param zooKeeperService
   *          referenced service
   */
  public void unsetZooKeeperService(final ZooKeeperService zooKeeperService) {
    if (_zkService == zooKeeperService) {
      _zkService = null;
      try {
        _zkConnection.disconnectZkSession();
      } catch (final IOException e) {
        _log.error("Cannot disconnect Zookeeper session.", e);
      } catch (final ClusterConfigException e) {
        _log.error("Cannot disconnect Zookeeper session.", e);
      }
      _zkConnection = null;
    }
  }

}
