/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.zookeeper.internal;

import java.io.File;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.zookeeper.server.PurgeTxnLog;

/**
 * Implements data file management (garbage collection) for the ZooKeeper's data files.
 */
public class ZooKeeperGC implements Runnable {

  /**
   * The number of snapshots to keep.
   */
  private final int _snapshotsToKeep;

  /**
   * Time to wait in ms between invocations of two garbage collection operations.
   */
  private final long _gcInterval;

  /**
   * Directory for snapshots.
   */
  private final File _dataDir;

  /**
   * Directory for logs.
   */
  private final File _logDir;

  /**
   * Flag for stopping ZooKeeperGC.
   */
  private boolean _stopped;

  /**
   * Local logger.
   */
  private final Log _log = LogFactory.getLog(getClass());

  /**
   * Constructor.
   * 
   * @param gcInterval
   *          time to wait in ms between invocations of two garbage collection operations
   * @param dataDir
   *          directory for snapshots
   * @param logDir
   *          directory for logs
   */
  public ZooKeeperGC(final long gcInterval, final String dataDir, final String logDir, final int snapshotsToKeep) {
    super();
    _gcInterval = gcInterval;
    _dataDir = new File(dataDir);
    _logDir = new File(logDir);
    _snapshotsToKeep = snapshotsToKeep;
    _stopped = false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void run() {
    synchronized (this) {
      while (!_stopped) {
        try {
          wait(_gcInterval);
        } catch (final InterruptedException e) {
          ; // Wake up!
        }
        if (!_stopped) {
          work();
        }
      }
    }
  }

  /**
   * Purges snapshots and logs keeping some last snapshots and the corresponding logs.
   */
  private void work() {
    try {
      _log.debug("GC started.");
      PurgeTxnLog.purge(_logDir, _dataDir, _snapshotsToKeep);
      _log.debug("GC done.");
    } catch (final Exception ex) {
      _log.warn("GC failed", ex);
    }
  }

  /**
   * Stops execution of ZooKeeperGC.
   */
  public synchronized void stop() {
    _stopped = true;
    notify();
  }
}
