/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.search;

import static org.apache.commons.lang.StringUtils.isBlank;

import java.util.UUID;

import org.apache.solr.common.params.CommonParams;
import org.apache.solr.common.params.MoreLikeThisParams;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.search.api.SearchService;
import org.eclipse.smila.search.api.helper.QueryBuilder;
import org.eclipse.smila.solr.SolrConstants;

/**
 * This class is used to build a Solr Search Record via API and execute the search/request.
 * 
 * @author pwissel
 * 
 */
public class SolrQueryBuilder extends QueryBuilder {

  /**
   * The default workflow.
   */
  private static final String DEFAULT_WORKFLOW = "SolrSearchDefaultWorkflow";

  /**
   * The default record id.
   */
  private static final String DEFAULT_RECORD_ID = "SolrSearchRecordId: ";

  /**
   * Default constructor.
   */
  public SolrQueryBuilder() {
    super(DEFAULT_WORKFLOW);
    final String id = DEFAULT_RECORD_ID + UUID.randomUUID();
    setId(id);
  }

  /**
   * Constructor.
   * 
   * @param workflowName
   *          the workflow workflow name.
   */
  public SolrQueryBuilder(String workflowName) {
    super(workflowName);

  }

  /**
   * Constructor.
   * 
   * @param workflowName
   *          the workflow name.
   * @param factory
   *          the data factory.
   */
  public SolrQueryBuilder(String workflowName, DataFactory factory) {
    super(workflowName, factory);
  }

  /**
   * Constructor.
   * 
   * @param workflowName
   *          the workflow name.
   * @param request
   *          the request record.
   */
  public SolrQueryBuilder(String workflowName, Record request) {
    super(workflowName, request);
  }

  /**
   * Set start parameter.
   * 
   * @param start
   *          the start parameter.
   * @return the solr query builder.
   */
  public SolrQueryBuilder setStart(int start) {
    setOffset(start);
    return this;
  }

  /**
   * Set rows parameter.
   * 
   * @param rows
   *          the rows parameter.
   * @return the solr query builder.
   */
  public SolrQueryBuilder setRows(int rows) {
    setMaxCount(rows);
    return this;
  }

  /**
   * Add fields parameter.
   * 
   * @param fields
   *          the fields parameter.
   * @return the solr query record.
   */
  public SolrQueryBuilder addFields(String... fields) {
    addResultAttributes(fields);
    return this;
  }

  /**
   * Set the request handler.
   * 
   * @param name
   *          The value given should start with a leading /. However, if it is missing it will be added. Blank values
   *          will result in removing the request handler parameter (reset to default).
   * @return the solr query builder.
   */
  public SolrQueryBuilder setRequestHandler(String name) {
    if (isBlank(name)) {
      getSolrQueryMap().remove(CommonParams.QT);
    } else {
      getSolrQueryMap().put(CommonParams.QT, name);
    }
    return this;
  }

  /**
   * Adds a native solr filter query.
   * 
   * @param filterQuery
   *          the filter query.
   * @return the solr query builder.
   */
  public SolrQueryBuilder addFilterQuery(String filterQuery) {
    if (filterQuery != null) {
      getSolrQueryMap().getSeq(SolrConstants.FILTER_QUERY, true).add(filterQuery);
    }
    return this;
  }

  /**
   * Get the solr query map _solr.query.
   * 
   * @return the solr query map.
   */
  AnyMap getSolrQueryMap() {
    return getMetadata().getMap(SolrConstants.QUERY_MAP, true);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.search.api.helper.QueryBuilder#executeRequest(org.eclipse.smila.search.api.SearchService)
   */
  @Override
  public SolrResultAccessor executeRequest(SearchService searchService) throws ProcessingException {
    super.executeRequest(searchService);
    return new SolrResultAccessor(getWorkflowName(), getQuery());
  }

  /**
   * Set the *:* as the query string to find all documents. Mainly used for testing. Only works when used with the right
   * (default) query handler.
   * 
   * @return SolrQueryBuilder.
   */
  public SolrQueryBuilder setQueryFindAll() {
    setQuery("*:*");
    return this;
  }

  /**
   * Set shards.
   * 
   * @param shards
   *          the shards.
   * @return SolrQueryBuilder.
   */
  public SolrQueryBuilder setShards(String... shards) {
    for (String shard : shards) {
      getSolrQueryMap().getSeq(SolrConstants.SHARDS, true).add(shard);
    }
    return this;
  }

  /**
   * This will add the parameters to the _solr.query map and name it {@link SolrConstants#MORE_LIKE_THIS}. If the map
   * dosnt contain a value {@code mlt} then it will be added setting it to true, thus turning on MLT.
   * 
   * @param mltArgs
   */
  public void setMoreLikeThis(AnyMap mltArgs) {
    getSolrQueryMap().put(SolrConstants.MORE_LIKE_THIS, mltArgs);
    final Value value = mltArgs.getValue(MoreLikeThisParams.MLT);
    if (value == null) {
      mltArgs.put(MoreLikeThisParams.MLT, true);
    }

  }

  /**
   * Adds the given name-value as a parameter to the native parameter map. Note, that
   * {@link AnyMap#add(String, org.eclipse.smila.datamodel.Any)} is used, which does some auto conversion to support
   * multiple values.
   * 
   * @see {@link AnyMap#add(String, org.eclipse.smila.datamodel.Any)}
   * 
   * @param parameterName
   * @param parameterValue
   */
  public void addNativeParam(String parameterName, Object parameterValue) {
    final AnyMap map = getSolrQueryMap().getMap(QueryConstants.NATIVE_PARAMETERS, true);
    map.add(parameterName, getFactory().autoConvertValue(parameterValue));
  }

  /**
   * Sets the given name-value pair, posisbly overriding an existing one under the same key.
   * 
   * @see #addNativeParam(String, Object)
   * @param parameterName
   *          the parameter name
   * @param parameterValue
   *          the parameter value
   */
  public void setNativeParam(String parameterName, Object parameterValue) {
    final AnyMap map = getSolrQueryMap().getMap(QueryConstants.NATIVE_PARAMETERS, true);
    map.put(parameterName, getFactory().autoConvertValue(parameterValue));
  }

  /**
   * @param filterGroupQ
   * @param create
   * @return
   */
  public AnyMap addFilterGroup(String filterGroupName) {
    return getSolrQueryMap().getMap(SolrConstants.FILTER_GROUPS, true).getMap(filterGroupName, true);
  }

  /**
   * Adds the given params to the local params map of the filter group.
   * 
   * @param filterGroupName
   *          the filter group q
   * @param params
   *          the objects are {@code AnyUtil.objectToAny(param);}
   * @return the local params map
   */
  public AnyMap addFilterGroupLocalParam(String filterGroupName, Object... params) {
    final AnyMap localParams = addFilterGroup(filterGroupName).getMap(SolrConstants.LOCAL_PARAMS, true);
    if (params != null) {
      for (Object param : params) {
        AnyUtil.objectToAny(param);
      }
    }

    return localParams;
  }

}
