/*******************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management Europe GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.persistence.zk;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;

/**
 * Zookeeper structure:
 * 
 * <pre>
 * /smila/jobmanager/jobs/&lt;job-name>/workflow-runs/&lt;workflow-run-id>/data/tasks/&lt;task-id>
 * ............................................................................/transient-bulks/&lt;subfolder>/&lt;bulk>
 * ............................................................................/barriers/&lt;action-position>/tasks
 * ........................................................................................................../bulks
 * ......................................................................./finishing
 *
 * ................................./data/&lt;data-node>              // job counters etc.
 * ................................./worker-data/&lt;workername>      // worker specific counters etc.
 * ................................./custom/&lt;workername>/&lt;path> // custom data stored by a worker during a job run
 * ................................./jobrun-definitions/              // definitions used in a job run
 * ..................................................../jobdef
 * ................................................... /wfdef
 * ..................................................../bucketdef/&lt;bucket-name>
 * ....................../buckets/&lt;bucket-id>/&lt;job-name>        // jobs triggered by bucket
 * </pre>
 * 
 */
public final class ZkPaths {
  /** prefix for jobmanager data. */
  public static final String JOBMANAGER_PREFIX = "/smila/jobmanager";

  /** workflow runs of a job run. */
  public static final String NODE_WORKFLOW_RUNS = "workflow-runs";

  /** data of a job run. */
  public static final String NODE_BUCKETS = "buckets";

  /** data of a job run. */
  public static final String NODE_JOBS = "jobs";

  /** global data of a job run. */
  public static final String NODE_DATA = "data";

  /** worker specific data of a job run. */
  public static final String NODE_WORKERDATA = "worker-data";

  /** tasks of a workflow step. */
  public static final String NODE_TASKS = "tasks";

  /** transient bulks of a workflow run. */
  public static final String NODE_TRANSIENT_BULKS = "transient-bulks";

  /** root node for stored definitions that are used by a job run. */
  public static final String NODE_RUN_DEFINITIONS = "jobrun-definitions";

  /** stores job definition used by a job run. */
  public static final String NODE_RUN_DEFINITIONS_JOB = "jobdef";

  /** stores workflow definition used by a job run. */
  public static final String NODE_RUN_DEFINITIONS_WORKFLOW = "wfdef";

  /** stores bucket definitions used by a job run. */
  public static final String NODE_RUN_DEFINITIONS_BUCKET = "bucketdef";

  /** indicates whether a workflow run is currently finished. */
  public static final String NODE_WORKFLOW_RUN_FINISHING = "finishing";

  /** root node for stored custom worker data. */
  public static final String NODE_CUSTOM = "custom";

  /**
   * @return root path for all stored jobs
   */
  public static String getJobsPath() {
    return JOBMANAGER_PREFIX + '/' + NODE_JOBS;
  }

  /**
   * @return root path for given job name
   */
  public static String getJobPath(final String jobName) {
    return getJobsPath() + '/' + jobName;
  }

  /**
   * @return root path where to store the definitions for the job (run).
   */
  public static String getJobRunDefinitionsPath(final String jobName) {
    return getJobPath(jobName) + '/' + NODE_RUN_DEFINITIONS;
  }

  /**
   * @return path where the job definition for the given job is stored.
   */
  public static String getJobRunDefPathForJobDef(final String jobName) {
    return getJobRunDefinitionsPath(jobName) + '/' + NODE_RUN_DEFINITIONS_JOB;
  }

  /**
   * @return path where the workflow definition for the given job is stored.
   */
  public static String getJobRunDefPathForWorkflowDef(final String jobName) {
    return getJobRunDefinitionsPath(jobName) + '/' + NODE_RUN_DEFINITIONS_WORKFLOW;
  }

  /**
   * @return path where the bucket definitions for the given job are stored.
   */
  public static String getJobRunDefPathForBucketDefs(final String jobName) {
    return getJobRunDefinitionsPath(jobName) + '/' + NODE_RUN_DEFINITIONS_BUCKET;
  }

  /**
   * @param jobName
   *          job name
   * @return path for job run data for given job.
   */
  public static String getJobDataPath(final String jobName) {
    return getJobPath(jobName) + '/' + NODE_DATA;
  }

  /**
   * @param jobName
   *          job name
   * @return root path for worker specific job run data.
   */
  public static String getJobWorkerDataPath(final String jobName) {
    return getJobPath(jobName) + '/' + NODE_WORKERDATA;
  }

  /**
   * @param jobName
   *          job name
   * @param workerName
   *          worker name
   * @return path for worker specific job run data.
   */

  public static String getJobWorkerDataPath(final String jobName, final String workerName) {
    return getJobWorkerDataPath(jobName) + '/' + workerName;
  }

  /**
   * @param jobName
   *          job name
   * @return root path for workflow runs for given job name.
   */
  public static String getWorkflowRunsPath(final String jobName) {
    return getJobPath(jobName) + '/' + NODE_WORKFLOW_RUNS;
  }

  /**
   * @param jobName
   *          job name
   * @param workflowRunId
   *          workflow run id
   * @return path for given job and workflow run id
   */
  public static String getWorkflowRunPath(final String jobName, final String workflowRunId) {
    return getWorkflowRunsPath(jobName) + '/' + workflowRunId;
  }

  /**
   * @return path where all data for given workflow run is stored.
   */
  public static String getWorkflowRunDataPath(final String jobName, final String workflowRunId) {
    return getWorkflowRunPath(jobName, workflowRunId) + "/data";
  }

  /**
   * @return path where task ids for given workflow run are stored.
   */
  public static String getWorkflowRunTasksPath(final String jobName, final String workflowRunId) {
    return getWorkflowRunDataPath(jobName, workflowRunId) + '/' + NODE_TASKS;
  }

  /**
   * @return path where all transient bulk ids for given workflow run are stored (in sub folders)
   */
  public static String getWorkflowRunBulksPath(final String jobName, final String workflowRunId) {
    return getWorkflowRunDataPath(jobName, workflowRunId) + '/' + NODE_TRANSIENT_BULKS;
  }

  /**
   * @return sub path where transient bulk ids for given workflow run are stored.
   */
  public static String getWorkflowRunBulksSubPath(final String jobName, final String workflowRunId,
    final int currentSubFolder) {
    return getWorkflowRunDataPath(jobName, workflowRunId) + '/' + NODE_TRANSIENT_BULKS + "/" + currentSubFolder;
  }

  /**
   * @return path where barrier data for the given workflow run is stored
   */
  public static String getWorkflowRunBarriersPath(final String jobName, final String workflowRunId,
    final int actionPosition) {
    return getWorkflowRunDataPath(jobName, workflowRunId) + "/barriers/" + actionPosition + '/';
  }

  /**
   * @return path where task ids are stored that must be finished before a barrier action is triggered.
   */
  public static String getWorkflowRunBarrierTasksPath(final String jobName, final String workflowRunId,
    final int actionPosition) {
    return getWorkflowRunBarriersPath(jobName, workflowRunId, actionPosition) + NODE_TASKS;
  }

  /**
   * @return path for task to be stored that must be finished before a barrier action is triggered.
   */
  public static String getWorkflowRunBarrierTaskPath(final String jobName, final String workflowRunId,
    final int actionPosition, final String taskId) {
    return getWorkflowRunBarrierTasksPath(jobName, workflowRunId, actionPosition) + '/' + taskId;
  }

  /**
   * @return path where bulk ids are stored that are used to trigger a barrier action
   */
  public static String getWorkflowRunBarrierBulksPath(final String jobName, final String workflowRunId,
    final int actionPosition) {
    return getWorkflowRunBarriersPath(jobName, workflowRunId, actionPosition) + "bulks";
  }

  /**
   * @return path for bulks to be stored for triggering a barrier task later.
   */
  public static String getWorkflowRunBarrierBulkPath(final String jobName, final String workflowRunId,
    final int actionPosition, final String bulkId) {
    return getWorkflowRunBarrierBulksPath(jobName, workflowRunId, actionPosition) + '/' + encode(bulkId);
  }

  /**
   * @return path where we set a mark to ensure that only one finishing tasks leads to opening the barrier.
   */
  public static String getWorkflowRunBarrierOpeningPath(final String jobName, final String workflowRunId,
    final int actionPosition) {
    return getWorkflowRunBarriersPath(jobName, workflowRunId, actionPosition) + "open";
  }

  /**
   * @param bucketId
   *          bucket Id
   * @return path for node containing job names triggered by this bucket.
   */
  public static String getTriggerBucketPath(final String bucketId) {
    return JOBMANAGER_PREFIX + '/' + NODE_BUCKETS + '/' + encode(bucketId);
  }

  /**
   * @param bucketId
   *          bucket Id
   * @param jobName
   *          job name
   * @return path for node describing that a node is triggered by a bucket
   */
  public static String getTriggeredJobPath(final String bucketId, final String jobName) {
    return getTriggerBucketPath(bucketId) + '/' + jobName;
  }

  /** @returns zk path to the custom data for the provided parameters. */
  public static String getCustomDataPath(final String job, final String worker, final String... path) {
    final StringBuilder customPath = new StringBuilder(getJobPath(job));
    customPath.append('/').append(NODE_CUSTOM).append('/').append(worker);
    for (final String pathElement : path) {
      customPath.append('/').append(pathElement);
    }
    return customPath.toString();
  }

  /**
   * @param value
   *          a string
   * @return URL encoded version.
   */
  public static String encode(final String value) {
    try {
      return URLEncoder.encode(value, "utf-8");
    } catch (final UnsupportedEncodingException ex) {
      throw new IllegalStateException("UTF-8 MUST BE SUPPORTED!", ex);
    }
  }

  /**
   * @param encodedValue
   *          an URL encoded string
   * @return decoded version.
   */
  public static String decode(final String encodedValue) {
    try {
      return URLDecoder.decode(encodedValue, "utf-8");
    } catch (final UnsupportedEncodingException ex) {
      throw new IllegalStateException("UTF-8 MUST BE SUPPORTED!", ex);
    }
  }

  private ZkPaths() {
    // no instantation, please.
  }

}
