/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Tobias Liefke - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.processing.pipelets;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;

/**
 * Sets an attribute value for every record.
 * 
 * @author Tobias Liefke
 */
public class SetValuePipelet implements Pipelet {

  /** Name of the property that contains attribute name. */
  private static final String PARAM_ATTRIBUTE = "outputAttribute";

  /** Name of the property that contains the value to set. */
  private static final String PARAM_VALUE = "value";

  /** Name of the property that indicates to overwrite existing values. */
  private static final String PARAM_OVERWRITE = "overwrite";

  /** The pipelet's configuration. */
  private AnyMap _config;

  /** Our logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** Save the configuration for later processing. */
  @Override
  public void configure(final AnyMap configuration) throws ProcessingException {
    _config = configuration;
  }

  /**
   * Set the value for all pipelets.
   */
  @Override
  public String[] process(final Blackboard blackboard, final String[] recordIds) throws ProcessingException {
    final ParameterAccessor paramAccessor = new ParameterAccessor(blackboard, _config);
    final ResultCollector resultCollector =
      new ResultCollector(paramAccessor, _log, ProcessingConstants.DROP_ON_ERROR_DEFAULT);
    for (final String id : recordIds) {
      paramAccessor.setCurrentRecord(id);
      // The attribute name
      final String outputAttribute = paramAccessor.getParameter(PARAM_ATTRIBUTE, null);
      // The value to set
      final Any value = paramAccessor.getParameterAny(PARAM_VALUE);
      if (value != null && outputAttribute != null) {
        try {
          final Any oldValue = blackboard.getMetadata(id).put(outputAttribute, value);
          if (oldValue != null && !paramAccessor.getBooleanParameter(PARAM_OVERWRITE, Boolean.FALSE)) {
            // Restore old value, if no overwrite parameter is set
            blackboard.getMetadata(id).put(outputAttribute, oldValue);
          }
          resultCollector.addResult(id);
        } catch (final Exception ex) {
          resultCollector.addFailedResult(id, ex);
        }
      }
    }
    return resultCollector.getResultIds();
  }

}
