/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client;

/**
 * Helper class providing resources for the standard SMILA REST API to be used in the RestClient. All resulting
 * resources will contain leading and trailing slashes.
 */
public class ResourceHelper {

  /** default resource root. */
  private static final String DEFAULT_ROOT = "/smila/";

  /** pipelets url fragment. */
  private static final String PIPELETS = "pipelets/";

  /** pipeline url fragment. */
  private static final String PIPELINES = "pipeline/";

  /** jobmanager url fragment. */
  private static final String JOB_MANAGER_BASE = "jobmanager/";

  /** dataobjecttypes url fragment. */
  private static final String DATA_OBJECT_TYPES = "dataobjecttypes/";

  /** buckets url fragment. */
  private static final String BUCKETS = "buckets/";

  /** workers url fragment. */
  private static final String WORKERS = "workers/";

  /** workflows url fragment. */
  private static final String WORKFLOWS = "workflows/";

  /** jobs url fragment. */
  private static final String JOBS = "jobs/";

  /** finish url fragment. */
  private static final String FINISH = "finish/";

  /** cancel url fragment. */
  private static final String CANCEL = "cancel/";

  /** job url fragment. */
  private static final String JOB = "job/";

  /** record url fragment. */
  private static final String RECORD = "record/";

  /** bulk url fragment. */
  private static final String BULK = "bulk/";

  /** tasks url fragment. */
  private static final String TASKS = "tasks/";

  /** inprogress type. */
  private static final String INPROGRESS = "inprogress";

  /** todo type. */
  private static final String TODO = "todo";

  /** importing url fragment. */
  private static final String IMPORTING = "importing/";

  /** delta url fragment. */
  private static final String DELTA = "delta/";

  /** visitedlinks url fragment. */
  private static final String VISITED_LINKS = "visitedlinks/";

  /** store url fragment. */
  private static final String STORE = "store/";

  /** The root context resource. Contains a final "/". */
  private final String _rootContextResource;

  /**
   * Create instance for default root context "/smila".
   */
  public ResourceHelper() {
    this(DEFAULT_ROOT);
  }

  /**
   * Creates instance with a specific root context.
   * 
   * @param rootContextResource
   *          the root context for the SMILA handlers. (E.g. "/smila").
   */
  public ResourceHelper(final String rootContextResource) {
    String tmp = rootContextResource;
    if (!tmp.startsWith("/")) {
      tmp = "/" + tmp;
    }
    if (!tmp.endsWith("/")) {
      tmp = tmp + "/";
    }
    this._rootContextResource = tmp;
  }

  /**
   * @return the resource for the pipelets.
   */
  public String getPipeletsRersource() {
    return this._rootContextResource + PIPELETS;
  }

  /**
   * @return the resource for the a given pipelet.
   */
  public String getPipeletResource(final String pipeletClassName) {
    return getPipeletsRersource() + pipeletClassName + "/";
  }

  /**
   * @return the resource for the pipelines.
   */
  public String getPipelinesResource() {
    return this._rootContextResource + PIPELINES;
  }

  /**
   * @return the resource for the a given pipeline.
   */
  public String getPipelineResource(final String pipelineName) {
    return getPipelinesResource() + pipelineName + "/";
  }

  /**
   * @return the base resource for the JobManager.
   */
  public String getJobManagerResource() {
    return this._rootContextResource + JOB_MANAGER_BASE;
  }

  /**
   * @return the resource for the DataObjectTypes.
   */
  public String getDataObjectTypesResource() {
    return getJobManagerResource() + DATA_OBJECT_TYPES;
  }

  /**
   * @return the resource for a given DataObjectType.
   */
  public String getDataObjectTypeResource(final String dataObjectTypeName) {
    return getDataObjectTypesResource() + dataObjectTypeName + "/";
  }

  /**
   * @return the resource for the Buckets.
   */
  public String getBucketsResource() {
    return getJobManagerResource() + BUCKETS;
  }

  /**
   * @return the resource for a given Bucket.
   */
  public String getBucketResource(final String bucketName) {
    return getBucketsResource() + bucketName + "/";
  }

  /**
   * @return the resource for the Workers.
   */
  public String getWorkersResource() {
    return getJobManagerResource() + WORKERS;
  }

  /**
   * @return the resource for a given Worker.
   */
  public String getWorkerResource(final String workerName) {
    return getWorkersResource() + workerName + "/";
  }

  /**
   * @return the resource for the Workflows.
   */
  public String getWorkflowsResource() {
    return getJobManagerResource() + WORKFLOWS;
  }

  /**
   * @return the resource for a given Workflow.
   */
  public String getWorkflowResource(final String workflowName) {
    return getWorkflowsResource() + workflowName + "/";
  }

  /**
   * @return the resource for the Jobs.
   */
  public String getJobsResource() {
    return getJobManagerResource() + JOBS;
  }

  /**
   * @return the resource for a given Job.
   */
  public String getJobResource(final String jobName) {
    return getJobsResource() + jobName + "/";
  }

  /**
   * @return the resource for a given Job Run.
   */
  public String getJobRunResource(final String jobName, final String jobRunId) {
    return getJobResource(jobName) + jobRunId + "/";
  }

  /**
   * @return the resource to use to finish a given Job Run.
   */
  public String getJobRunFinishResource(final String jobName, final String jobRunId) {
    return getJobRunResource(jobName, jobRunId) + FINISH;
  }

  /**
   * @return the resource to use to cancel a given Job Run.
   */
  public String getJobRunCancelResource(final String jobName, final String jobRunId) {
    return getJobRunResource(jobName, jobRunId) + CANCEL;
  }

  /**
   * @return the base resource for the Bulkbuilder push API for a given Job.
   */
  public String getPushToJobBaseResource(final String jobName) {
    return this._rootContextResource + JOB + jobName + "/";
  }

  /**
   * @return the resource for the Bulkbuilder record push API for a given Job.
   */
  public String getPushRecordToJobResource(final String jobName) {
    return getPushToJobBaseResource(jobName) + RECORD;
  }

  /**
   * @return the resource for the Bulkbuilder bulk push API for a given Job.
   */
  public String getPushBulkToJobResource(final String jobName) {
    return getPushToJobBaseResource(jobName) + BULK;
  }

  /**
   * @return the resource for the Task queues.
   */
  public String getTasksResource() {
    return this._rootContextResource + TASKS;
  }

  /**
   * @return the resource for the Task queue for a given type of a given Worker.
   */
  public String getTasksForWorkerResource(final String workerName, final String taskType) {
    return getTasksResource() + workerName + "/" + taskType + "/";
  }

  /**
   * @return the resource for the "inprogress" Task queue for a given Worker.
   */
  public String getTasksInProgressForWorkerResource(final String workerName) {
    return getTasksForWorkerResource(workerName, INPROGRESS);
  }

  /**
   * @return the resource for the "todo" Task queue for a given Worker.
   */
  public String getTasksToDoForWorkerResource(final String workerName) {
    return getTasksForWorkerResource(workerName, TODO);
  }

  /**
   * @return the resource for a given "inprogress" Task for a given Worker.
   */
  public String getTaskInProgressResource(final String workerName, final String taskName) {
    return getTasksForWorkerResource(workerName, INPROGRESS) + taskName + "/";
  }

  /**
   * @return the resource for a given "todo" Task for a given Worker.
   */
  public String getTaskToDoResource(final String workerName, final String taskName) {
    return getTasksForWorkerResource(workerName, TODO) + taskName + "/";
  }

  /**
   * @return the base resource for the importing framework.
   */
  public String getImportingResource() {
    return this._rootContextResource + IMPORTING;
  }

  /**
   * @return the resource for the delta service of the importing framework.
   */
  public String getDeltaServicesResource() {
    return getImportingResource() + DELTA;
  }

  /**
   * @return the resource the delta service for a given source.
   */
  public String getDeltaServiceResource(final String sourceName) {
    return getDeltaServicesResource() + sourceName + "/";
  }

  /**
   * @return the resource for the visited links service of the importing framework.
   */
  public String getVisitedLinksResource() {
    return getImportingResource() + VISITED_LINKS;
  }

  /**
   * @return the resource the visited links service for a given source.
   */
  public String getVisitedLinksResource(final String sourceName) {
    return getVisitedLinksResource() + sourceName + "/";
  }

  /**
   * @return the resource for the object store service.
   */
  public String getStoresResource() {
    return this._rootContextResource + STORE;
  }

  /**
   * @return the resource for a given object store.
   */
  public String getStoreResource(final String storeName) {
    return getStoresResource() + storeName + "/";
  }

  /**
   * @return the resource for a given object in a given store.
   */
  public String getStoreObjectResource(final String storeName, final String objectId) {
    return getStoreResource(storeName) + objectId + "/";
  }
}
