/*******************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.processing.httphandler;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.PipeletTracker;
import org.eclipse.smila.processing.ProcessingException;

/**
 * Implements the handling of HTTP requests by Pipelets.
 */
public class PipeletProcessHandler extends AProcessHandler {

  /** associated pipelet tracker */
  private PipeletTracker _tracker;

  /**
   * {@inheritDoc}
   * 
   * @throws BlackboardAccessException
   * @throws ProcessingException
   */
  @Override
  public String[] doProcess(final String pipeletName, final Blackboard blackboard, final String[] recordIds)
    throws BlackboardAccessException, ProcessingException {
    final Pipelet pipelet = createPipelet(pipeletName);
    configurePipelet(pipelet, blackboard.getRecord(recordIds[0]));
    return pipelet.process(blackboard, recordIds);
  }

  private Pipelet createPipelet(final String pipeletName) {
    final Class<? extends Pipelet> pipeletClass = _tracker.getRegisteredPipelets().get(pipeletName);
    if (pipeletClass == null) {
      throw new IllegalArgumentException("Unknown pipelet class " + pipeletName);
    }
    try {
      return pipeletClass.newInstance();
    } catch (InstantiationException | IllegalAccessException ex) {
      throw new IllegalArgumentException("Could not create pipelet of class " + pipeletName, ex);
    }
  }

  private void configurePipelet(final Pipelet pipelet, final Record inputRecord) {
    final Any configuration = inputRecord.getMetadata().get("_configuration");
    if (configuration != null && configuration.isMap()) {
      try {
        pipelet.configure(configuration.asMap());
      } catch (final ProcessingException ex) {
        throw new IllegalArgumentException("Invalid configuration for pipelet of class "
          + pipelet.getClass().getName(), ex);
      }
    }
  }

  /** set pipelet tracker reference (used by DS). */
  public void setPipeletTracker(final PipeletTracker tracker) {
    _tracker = tracker;
  }

  /** remove pipelet tracker reference (used by DS). */
  public void unsetPipeletTracker(final PipeletTracker tracker) {
    if (_tracker == tracker) {
      _tracker = null;
    }
  }

}
