/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager.persistence;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCounter;
import org.eclipse.smila.taskmanager.TaskList;
import org.eclipse.smila.taskmanager.TaskmanagerException;

/**
 * Implementors of this interface implement the storage of tasks in so-called workerNames which can be seen as virtual
 * queues. There may be but needn't to be a physical queue representation for a concrete worker.
 * 
 * Received tasks are not removed from the workerName directly, but marked as "in process". Tasks "in process" will not
 * be redelivered, as long as they aren't set unprocessed.
 * 
 * A client should surround consecutive calls with {@link #startSession()} and {@link #endOfSession(String)} calls to
 * get better performance.
 */
public interface TaskStorage {

  /**
   * Stores a new task to the todo queue of the given task's worker.
   * 
   * @param task
   *          the task to store, contains the worker name
   * @throws TaskmanagerException
   *           if send fails
   */
  void storeTask(Task task) throws TaskmanagerException;

  /**
   * Stores a new task to the todo queue of the given workerName. The worker name stored in the task will be ignored.
   * 
   * @param task
   *          the task to store.
   * @param workerName
   *          the worker for which to store the task.
   * @throws TaskmanagerException
   *           if send fails
   */
  void storeTask(String workerName, Task task) throws TaskmanagerException;

  /**
   * Stores a new task to the inprogress queue of the given task's worker.
   * 
   * @param task
   *          the task to store, contains the worker name
   * @throws TaskmanagerException
   *           if send fails
   */
  void storeInProgressTask(Task task) throws TaskmanagerException;

  /**
   * Returns a task that's not "in process" from the given workerName for the given messageCondition. The returned task
   * will be marked as "in process".
   * 
   * 
   * @param workerName
   *          the workerName where to get the task from.
   * @param workerHost
   *          the host name where the worker is running that requested the task
   * @param qualifiers
   *          if not null, allowed qualifiers of returned task.
   * @return a task from the workerName for the given messageCondition (if any) or <code>null</code> if there is none
   *         available.
   * @throws TaskmanagerException
   *           if receive fails
   */
  Task getTask(String workerName, String workerHost, Collection<String> qualifiers) throws TaskmanagerException;

  /**
   * read the task content of an in-progress task.
   * 
   * @param workerName
   *          name of worker
   * @param taskId
   *          task Id
   * @return task contents or null if no such task exists.
   * @throws TaskmanagerException
   *           if task is not in-progress or other errors.
   */
  Task getInProgressTask(String workerName, String taskId) throws TaskmanagerException;

  /**
   * Remove the task from the inprogress part of the workers queue.
   * 
   * @param workerName
   *          the workerName from where the task was originally get.
   * @param taskId
   *          the task identifier.
   * @throws TaskmanagerException
   *           if taskId is invalid or not marked as finishing
   */
  void deleteTask(String workerName, String taskId) throws TaskmanagerException;

  /**
   * Signals that the worker processing the given task is still alive.
   * 
   * @param workerName
   *          name/id as string
   * @param taskId
   *          task id that is currently processed as string
   * @throws TaskmanagerException
   *           if any error occurs
   */
  void keepAlive(String workerName, String taskId) throws TaskmanagerException;

  /**
   * Remove all tasks from the given task pipe.
   * 
   * @param workerName
   *          name/id as string
   * @throws TaskmanagerException
   *           if any error occurs
   */
  void purge(String workerName) throws TaskmanagerException;

  /**
   * prevent new todo tasks from being added for the given worker and qualifiers. Used by
   * org.eclipse.smila.taskmanager.TaskManager#finishTasks to prevent tasks (re-)entering the cleared qualifiers for the
   * some time. The locks should be kept active by the TaskStorage implementation for at least some minutes, afterwards
   * they may be cleaned up.
   * 
   * @param workerName
   *          a worker's name
   * @param qualifiers
   *          a list of qualifiers to lock
   */
  void lockQualifiers(String workerName, Collection<String> qualifiers);

  /**
   * check if a lock exists for given worker and qualifier.
   */
  boolean isLockedQualifier(final String workerName, final String qualifier) throws TaskmanagerException;

  /**
   * Get number of tasks in all current queues.
   * 
   * @return task counters
   * @throws TaskmanagerException
   *           counting fails
   */
  Map<String, TaskCounter> getTaskCounters() throws TaskmanagerException;

  /**
   * Get task list for current task pipe and sub-type.
   * 
   * @param workerName
   *          name/id as string
   * @param section
   *          task pipe sub-type: "inprogress" or "todo".
   * @param maxCount
   *          max. number of tasks in return list
   * @return task list
   * @throws TaskmanagerException
   *           if accessing task list fails
   */
  TaskList getTaskList(final String workerName, final String section, final int maxCount)
    throws TaskmanagerException;

  /**
   * Prepares information on a task stored in the task storage.
   * 
   * @param workerName
   *          pipe in which the task is stored
   * @param section
   *          queue section ("todo" or "inprogress")
   * @param taskName
   *          task identifier
   * @return task information
   * 
   * @throws TaskmanagerException
   *           on error
   */
  Any getTaskInfo(final String workerName, final String section, final String taskName) throws TaskmanagerException;

  /**
   * @return number of nodes that may fail before tasks get lost.
   */
  long getFailSafetyLevel();

  /**
   * remove all tasks from storage and reset all counters.
   * 
   * @throws TaskmanagerException
   *           error
   */
  void clear() throws TaskmanagerException;

  /**
   * @param maxTasks
   *          maximum number of tasks that should be delivered to a host. (scale up control)
   */
  void setMaxNoOfTasksPerHost(long maxTasks);

  /**
   * @return maximum number of tasks that should be delivered to a host. (scale up control)
   */
  long getMaxNoOfTasksPerHost();

  /**
   * @return number of tasks in progress on different hosts.
   * @throws TaskmanagerException
   *           on error.
   */
  Map<String, Integer> getScaleUpCounters() throws TaskmanagerException;

  /**
   * Remove canceled tasks identified by the filter map from todo and in-progress queue.
   * 
   * @param filterMap
   *          map to identify tasks to be removed
   * @throws TaskmanagerException
   *           an exception if something went wrong
   */
  void removeTasks(final AnyMap filterMap) throws TaskmanagerException;

  /**
   * @param name
   *          the name of the task queue to create.
   * @throws TaskmanagerException
   */
  void createTaskQueue(final String name) throws TaskmanagerException;

  /**
   * @param name
   *          the task queue name to check
   * @return 'true' if task for given name exists, 'false' otherwise.
   * @throws TaskmanagerException
   */
  boolean hasTaskQueue(final String name) throws TaskmanagerException;

  /**
   * return new list of tasks with only those tasks from the given task list
   * <ul>
   * <li>that do not have property {@link Task#PROPERTY_UNIQUENESS_TAG} set or
   * <li>for which currently no task is in the Todo queue that has the same uniqueness tag.
   * </ul>
   * 
   * @param tasksToBeAdded
   *          tasks created as follow-up tasks but not yet added to the task manager
   * @return tasks that are not already contained in its task queue's todo list
   */
  List<Task> filterDuplicates(final List<Task> tasksToBeAdded) throws TaskmanagerException;
}
