/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.definitions;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.InvalidValueTypeException;

/**
 * Definition of a job. JSON format:
 * 
 * <pre>
 * { 
 *   "name": "optimizeWikipediaIndex", 
 *   "parameters":
 *     { 
 *       "indexName" : "wikipedia",
 *       "mergeParameters" : "..."
 *     },
 *   "workflow" : "indexOptimize"
 * }
 * </pre>
 * 
 * "parameters" are optional but must not be "null".
 */
public class JobDefinition extends DefinitionBase {

  /** name of JSON property for lists of jobs. */
  public static final String KEY_JOBS = "jobs";

  /** name of JSON property for job modes list. */
  public static final String KEY_MODES = "modes";

  /** name of JSON property for job parameters map. */
  public static final String KEY_PARAMETERS = "parameters";

  /** name of JSON property for job workflow. */
  public static final String KEY_WORKFLOW = "workflow";

  /** name of JSON property for job task control. */
  public static final String KEY_TASK_CONTROL = "taskControl";

  /** name of JSON property for job task control delay. */
  public static final String KEY_TASK_CONTROL_WORKERS = "workers";

  /** name of JSON property for job task control delay. */
  public static final String KEY_TASK_CONTROL_DELAY = "delay";

  /** optional list of allowed job run modes. */
  private final List<JobRunMode> _modes;

  /** parameters are optional. */
  private final AnyMap _parameters;

  /** workflow is required. */
  private final String _workflow;

  /** name of workers for which tasks should be delayed. */
  private final List<String> _taskDelayWorkers = new ArrayList<String>();

  /** delay for task-delay workers. */
  private long _taskDelay;

  /**
   * parse job definition from Any object.
   * 
   * @param job
   *          see class comment for format of Any object
   * @throws InvalidDefinitionException
   *           error parsing object.
   */
  public JobDefinition(final AnyMap job) throws InvalidDefinitionException {
    super(job);
    _workflow = AccessAny.getStringRequired(job, KEY_WORKFLOW);
    final Any parametersAny = job.get(KEY_PARAMETERS);
    if (parametersAny == null) {
      _parameters = null;
    } else if (!parametersAny.isMap()) {
      throw new InvalidDefinitionException("Error reading jobs record object. Field parameter is not of type MAP.");
    } else {
      try {
        _parameters = (AnyMap) parametersAny;
      } catch (final Exception e) {
        throw new InvalidDefinitionException("Error reading parameters for jobs record object.", e);
      }
    }
    final AnySeq modesAny = job.getSeq(KEY_MODES);
    try {
      _modes = JobRunMode.parseModes(modesAny);
    } catch (final InvalidDefinitionException ex) {
      throw new InvalidDefinitionException("Job '" + _name + "': " + ex.getMessage());
    }
    final AnyMap taskControl = job.getMap(KEY_TASK_CONTROL);
    if (taskControl != null) {
      parseTaskControl(taskControl);
    }
  }

  /** parse values from task control map. */
  private void parseTaskControl(final AnyMap taskControl) throws InvalidDefinitionException {
    try {
      final AnySeq workers = taskControl.getSeq(KEY_TASK_CONTROL_WORKERS);
      if (workers == null || workers.isEmpty()) {
        throw new InvalidDefinitionException("Job '" + _name
          + "': task control section does not contain worker names.");
      }
      final Long delayValue = taskControl.getLongValue(KEY_TASK_CONTROL_DELAY);
      if (delayValue == null) {
        throw new InvalidDefinitionException("Job '" + _name
          + "': task control section does not contain value for delay.");
      }
      if (delayValue < 0) {
        throw new InvalidDefinitionException("Job '" + _name + "': task control delay must not be negative.");
      }
      for (final Any any : workers) {
        _taskDelayWorkers.add(any.asValue().asString());
      }
      _taskDelay = delayValue;
    } catch (final InvalidValueTypeException ex) {
      throw new InvalidDefinitionException("Job '" + _name + "' contains invalid values in task control section: "
        + ex.getMessage());
    }
  }

  /**
   * Parse on single job from an Any object containing a job description:
   * 
   * <pre>
   * {
   * // see class comment for job format.
   * }
   * </pre>
   * 
   * @param jobAny
   *          job as Any.
   * @return parsed job definition.
   * @throws InvalidDefinitionException
   *           error parsing Any.
   */
  public static JobDefinition parseJob(final AnyMap jobAny) throws InvalidDefinitionException {
    JobDefinition job = null;
    try {
      job = new JobDefinition(jobAny);
    } catch (final InvalidDefinitionException ex) {
      throw ex;
    } catch (final Exception ex) {
      throw new InvalidDefinitionException("Invalid any structure", ex);
    }
    return job;
  }

  /** get allowed job run modes for this job: either the configured ones or null. */
  public List<JobRunMode> getJobRunModes() {
    return _modes;
  }

  /** return the default job run mode for this job: either the first of the configured list of modes, or null. */
  public JobRunMode getDefaultJobRunMode() {
    if (_modes == null) {
      return null;
    }
    return getJobRunModes().get(0);
  }

  /**
   * @return the _parameters
   */
  public AnyMap getParameters() {
    return _parameters;
  }

  /**
   * @return the _workflow
   */
  public String getWorkflow() {
    return _workflow;
  }

  /** @return the names of workers for which tasks should be delivered with delay. */
  public List<String> getTaskDelayWorkers() {
    return _taskDelayWorkers;
  }

  /** @return the configured delay for task-delay workers. */
  public long getTaskDelay() {
    return _taskDelay;
  }

  /**
   * Returns the object as an Any representation. If "parameters" are set to null it will not be guaranteed if the
   * "parameters" property will be present in the resulting Any with a value of null or if it will not be present at
   * all.
   * 
   * @param includingAdditionalAttributes
   *          'true' if also any additional information in the AnyMap should be returned, 'false' if only the (minimal
   *          set of) relevant information should be returned.
   * @return Any object describing this job definition.
   */
  @Override
  public AnyMap toAny(final boolean includingAdditionalAttributes) {
    try {
      final AnyMap jobAny = super.toAny(includingAdditionalAttributes);
      if (!includingAdditionalAttributes) {
        jobAny.put(KEY_WORKFLOW, _workflow);
        if (_modes != null) {
          jobAny.put(KEY_MODES, JobRunMode.toAny(_modes));
        }
        if (_parameters != null) {
          jobAny.put(KEY_PARAMETERS, _parameters);
        }
        if (!_taskDelayWorkers.isEmpty()) {
          final AnySeq workerSeq = jobAny.getMap(KEY_TASK_CONTROL, true).getSeq(KEY_TASK_CONTROL_WORKERS, true);
          for (final String worker : _taskDelayWorkers) {
            workerSeq.add(worker);
          }
          jobAny.getMap(KEY_TASK_CONTROL).put(KEY_TASK_CONTROL_DELAY, _taskDelay);
        }
      }
      return jobAny;
    } catch (final Exception ex) {
      throw new IllegalStateException("Failed to create Any object for job " + _name, ex);
    }
  }
}
