/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.zookeeper.internal;

import java.io.File;
import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.zookeeper.server.NIOServerCnxnFactory;
import org.apache.zookeeper.server.ServerCnxnFactory;
import org.apache.zookeeper.server.ServerConfig;
import org.apache.zookeeper.server.ServerStats;
import org.apache.zookeeper.server.ZKDatabase;
import org.apache.zookeeper.server.ZooKeeperServer;
import org.apache.zookeeper.server.persistence.FileTxnLog;
import org.apache.zookeeper.server.persistence.FileTxnSnapLog;
import org.apache.zookeeper.server.quorum.QuorumPeer;
import org.apache.zookeeper.server.quorum.QuorumPeerConfig;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;

/**
 * Start a ZooKeeper stand-alone or cluster server and restart it after unexpected shutdowns.
 */
public class ZooKeeperServerRunner implements Runnable {

  /**
   * delay for restart after a server has ended with an exception.
   */
  private static final int RESTART_DELAY_AFTER_ERROR = 1000;

  /**
   * local logger.
   */
  private final Log _log = LogFactory.getLog(getClass());

  /**
   * true if running in a real multi-node cluster.
   */
  private final boolean _isCluster;

  /**
   * My index in cluster node list.
   */
  private final long _myId;

  /**
   * configuration for cluster peer.
   */
  private final QuorumPeerConfig _quorumPeerConfig;

  /**
   * configuration for stand-alone server.
   */
  private final ServerConfig _serverConfig;

  /**
   * cluster peer, if running in a real cluster.
   */
  private QuorumPeer _quorumPeer;

  /**
   * stand-alone ZooKeeper server, if running on single node.
   */
  private ZooKeeperServer _zkServer;

  /**
   * connection factory. Need to stop it for shutdown of stand-alone server, too. For cluster peers, the QuorumPeer
   * handles shutdown of factory itself.
   */
  private ServerCnxnFactory _cnxnFactory;

  /**
   * set by service deactivation, shutdown and do not restart the ZK server.
   */
  private boolean _doShutdown;

  /**
   * create server runner.
   * 
   * @param quorumPeerConfig
   *          configuration
   * @param isCluster
   *          cluster mode
   * @param myId
   *          index in cluster node list.
   */
  public ZooKeeperServerRunner(final QuorumPeerConfig quorumPeerConfig, final boolean isCluster, final long myId) {
    super();
    _isCluster = isCluster;
    _myId = myId;
    _quorumPeerConfig = quorumPeerConfig;
    _serverConfig = new ServerConfig();
    _serverConfig.readFrom(_quorumPeerConfig);
  }

  /**
   * run and restart server, as long as shutdown is not required from outside.
   */
  @Override
  public void run() {
    setDoShutdown(false);
    while (!doShutdown()) {
      try {
        cleanup();
        if (_isCluster) {
          runPeer();
        } else {
          runServer();
        }
        if (!doShutdown()) {
          _log.warn("ZooKeeper server has finished, will be restarted immediately.");
        }
      } catch (final Exception ex) {
        _log.warn("Error running ZooKeeper server, will retry to start in a moment.", ex);
        try {
          Thread.sleep(RESTART_DELAY_AFTER_ERROR);
        } catch (final Exception ex2) {
          _log.debug("Interrupt while waiting to restart failed server.", ex2);
        }
      }
    }
  }

  private synchronized void cleanup() {
    _quorumPeer = null;
    _zkServer = null;
    _cnxnFactory = null;
  }

  /**
   * start a stand alone server.
   * 
   * @throws IOException
   *           error creating the server
   * @throws InterruptedException
   *           error creating or starting the server.
   */
  private void runServer() throws IOException, InterruptedException {
    // during the tests we run with 100K prealloc in the logs.
    // on windows systems prealloc of 64M was seen to take ~15seconds
    // resulting in test failure (client timeout on first session).
    // set env and directly in order to handle static init/gc issues
    synchronized (this) {
      System.setProperty("zookeeper.preAllocSize", "100");
      final int fileTxnLogPreAllocSize = 100 * 1024;
      FileTxnLog.setPreallocSize(fileTxnLogPreAllocSize);
      _zkServer = new ZooKeeperServer();
      final FileTxnSnapLog ftxn =
        new FileTxnSnapLog(new File(_serverConfig.getDataLogDir()), new File(_serverConfig.getDataDir()));
      _zkServer.setTxnLogFactory(ftxn);
      _zkServer.setTickTime(_serverConfig.getTickTime());
      _zkServer.setMinSessionTimeout(_serverConfig.getMinSessionTimeout());
      _zkServer.setMaxSessionTimeout(_serverConfig.getMaxSessionTimeout());
      _cnxnFactory =
        NIOServerCnxnFactory.createFactory(_serverConfig.getClientPortAddress(), _serverConfig.getMaxClientCnxns());
      // new NIOServerCnxnFactory(_serverConfig.getClientPortAddress(), _serverConfig.getMaxClientCnxns());
      _cnxnFactory.startup(_zkServer);
    }
    try {
      _cnxnFactory.join();
    } finally {
      if (_zkServer.isRunning()) {
        _zkServer.shutdown();
      }
      _cnxnFactory.shutdown();
    }
  }

  /**
   * start a peer in a ZooKeeper cluster.
   * 
   * @throws IOException
   *           error creating the peer
   * @throws InterruptedException
   *           error starting the server
   */
  private void runPeer() throws IOException, InterruptedException {
    synchronized (this) {
      _log.info("Starting ZooKeeper quorum peer");
      _cnxnFactory =
        NIOServerCnxnFactory.createFactory(_quorumPeerConfig.getClientPortAddress(),
          _quorumPeerConfig.getMaxClientCnxns());

      _quorumPeer = new QuorumPeer();
      _quorumPeer.setClientPortAddress(_quorumPeerConfig.getClientPortAddress());
      _quorumPeer.setTxnFactory(new FileTxnSnapLog(new File(_quorumPeerConfig.getDataLogDir()), new File(
        _quorumPeerConfig.getDataDir())));

      _quorumPeer.setQuorumPeers(_quorumPeerConfig.getServers());
      _quorumPeer.setMyid(_myId);

      _quorumPeer.setElectionType(_quorumPeerConfig.getElectionAlg());
      _quorumPeer.setTickTime(_quorumPeerConfig.getTickTime());
      _quorumPeer.setMinSessionTimeout(_quorumPeerConfig.getMinSessionTimeout());
      _quorumPeer.setMaxSessionTimeout(_quorumPeerConfig.getMaxSessionTimeout());
      _quorumPeer.setInitLimit(_quorumPeerConfig.getInitLimit());
      _quorumPeer.setSyncLimit(_quorumPeerConfig.getSyncLimit());
      _quorumPeer.setQuorumVerifier(_quorumPeerConfig.getQuorumVerifier());
      _quorumPeer.setCnxnFactory(_cnxnFactory);
      _quorumPeer.setZKDatabase(new ZKDatabase(_quorumPeer.getTxnFactory()));
      _quorumPeer.setLearnerType(_quorumPeerConfig.getPeerType());
      _quorumPeer.start();
    }
    _quorumPeer.join();
  }

  /**
   * require to shutdown.
   */
  public synchronized void shutdown() {
    setDoShutdown(true);
    if (_quorumPeer != null) {
      try {
        _quorumPeer.shutdown();
      } catch (final Exception ex) {
        _log.warn("Error during shutdown of peer server", ex);
      }
    }
    if (_zkServer != null) {
      try {
        _zkServer.shutdown();
      } catch (final Exception ex) {
        _log.warn("Error during shutdown of stand-alone server, continuing to shutdown connection factory", ex);
      }
      try {
        _cnxnFactory.shutdown();
      } catch (final Exception ex) {
        _log.warn("Error during shutdown of connection factory of stand-alone server", ex);
      }
    }
  }

  private synchronized void setDoShutdown(final boolean doShutdown) {
    _doShutdown = doShutdown;
  }

  private synchronized boolean doShutdown() {
    return _doShutdown;
  }

  /**
   * @return string representing connection to the leader if there is one and we are not the leader.
   */
  synchronized String getRemoteLeaderConnection() {
    if (_quorumPeer != null) {
      final String[] peers = _quorumPeer.getQuorumPeers();
      if (peers == null) {
        return null;
      } else if (peers.length > 1) {
        // I am the leader (see impl. of getQuorumPeers())
        return null;
      } else if (peers.length > 0 && _quorumPeer.getQuorumPeers() != null
        && _quorumPeer.getQuorumPeers().length > 0) {
        return _quorumPeer.getQuorumPeers()[0];
      }
    }
    return null;
  }

  /**
   * @return zookeeper server statistics.
   */
  AnyMap getServerStatistics() {
    final AnyMap stats = DataFactory.DEFAULT.createAnyMap();
    stats.put("dataDir", _serverConfig.getDataDir());
    stats.put("dataLogDir", _serverConfig.getDataLogDir());
    ZooKeeperServer localServer = _zkServer; // we are running standalone
    if (_quorumPeer != null) {
      localServer = _quorumPeer.getActiveServer(); // we are running in cluster
    }
    if (localServer != null) {
      if (localServer.getZKDatabase() != null) {
        stats.put("nodeCount", localServer.getZKDatabase().getNodeCount());
        if (localServer.getZKDatabase().getDataTree() != null) {
          stats.put("watchCount", localServer.getZKDatabase().getDataTree().getWatchCount());
        }
        if (localServer.getZKDatabase().getSessions() != null) {
          stats.put("sessionsCount", localServer.getZKDatabase().getSessions().size());
        }
      }
      final ServerStats serverStats = localServer.serverStats();
      if (serverStats != null) {
        stats.put("avgLatency", serverStats.getAvgLatency());
        stats.put("minLatency", serverStats.getMinLatency());
        stats.put("maxLatency", serverStats.getMaxLatency());
        stats.put("outstandingRequests", serverStats.getOutstandingRequests());
        stats.put("packetsReceived", serverStats.getPacketsReceived());
        stats.put("packetsSent", serverStats.getPacketsSent());
      }
      stats.put("minSessionTimeout", localServer.getMinSessionTimeout());
      stats.put("maxSessionTimeout", localServer.getMaxSessionTimeout());
      stats.put("requestsInProcess", localServer.getInProcess());
    }
    return stats;
  }

  /**
   * @return this zookeeper server's state.
   */
  String getServerState() {
    if (_quorumPeer != null) {
      return _quorumPeer.getServerState();
    }
    return "standalone";
  }
}
