/***********************************************************************************************************************
 * Copyright (c) 2009 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Thomas Menzel (brox) - initial API and implementation
 **********************************************************************************************************************/
package org.eclipse.smila.search.servlet;

import static org.apache.commons.lang.StringUtils.isBlank;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.xml.XmlSerializationUtils;
import org.eclipse.smila.search.api.SearchService;
import org.eclipse.smila.search.servlet.activator.Activator;

/**
 * The RecordSearchServlet class.
 * 
 * @author pwissel
 * 
 */
public class RecordSearchServlet extends HttpServlet {

  /**
   * Default serial version UID.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Parameter file.
   */
  private static final String PARAM_FILE = "file";

  /**
   * Default search pipeline.
   */
  private static final String DEFAULT_SEARCH_PIPELINE = "SearchPipeline";

  /**
   * The log.
   */
  protected final Log _log = LogFactory.getLog(this.getClass());

  /**
   * {@inheritDoc}
   * 
   * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest,
   *      javax.servlet.http.HttpServletResponse)
   */
  @Override
  protected void doGet(final HttpServletRequest req, final HttpServletResponse resp) throws ServletException,
    IOException {
    processRequest(req, resp);
  }

  /**
   * {@inheritDoc}
   * 
   * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest,
   *      javax.servlet.http.HttpServletResponse)
   */
  @Override
  protected void doPost(final HttpServletRequest req, final HttpServletResponse resp) throws ServletException,
    IOException {
    processRequest(req, resp);
  }

  /**
   * Process the request.
   * 
   * @param req
   *          the request.
   * @param resp
   *          the response.
   * @throws IOException
   *           IOException.
   */
  private void processRequest(final HttpServletRequest req, final HttpServletResponse resp) throws IOException {
    try {

      final SearchService service = Activator.getInstance().getSearchService();
      if (service == null) {
        throw new IllegalStateException("SearchService must not be null.");
      }

      final String parameter = req.getParameter(PARAM_FILE);
      final InputStream inputStream;
      if (parameter == null) {
        inputStream = req.getInputStream();
      } else {
        inputStream = FileUtils.openInputStream(new File(parameter).getAbsoluteFile());
      }

      final Record searchRecord = XmlSerializationUtils.deserializeNoAttachments(inputStream);

      String workflow = searchRecord.getMetadata().getStringValue("_workflow");
      if (isBlank(workflow)) {
        workflow = DEFAULT_SEARCH_PIPELINE;
      }

      final String resultXml = service.searchAsXmlString(workflow, searchRecord);

      resp.setContentType("text/xml");
      final ServletOutputStream out = resp.getOutputStream();
      out.write(resultXml.getBytes("UTF-8"));

    } catch (final IllegalStateException exception) {
      handleError(HttpServletResponse.SC_CONFLICT, exception, resp);
    } catch (final Exception exception) {
      handleError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, exception, resp);
    }
  }

  /**
   * Handle an error.
   * 
   * @param errorCode
   *          the error code.
   * @param exception
   *          the exception.
   * @param resp
   *          the response.
   * @throws IOException
   *           IOException.
   */
  private void handleError(final int errorCode, final Throwable exception, final HttpServletResponse resp)
    throws IOException {
    final String msg = ExceptionUtils.getFullStackTrace(exception);
    _log.error(msg);
    resp.sendError(errorCode, msg);
  }

}
