/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.test;

import java.io.File;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.DeleteMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.PutMethod;
import org.apache.commons.io.FileUtils;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.workspace.WorkspaceHelper;

/**
 * Testcase for HttpHandler.
 * 
 */
public class TestBulkbuilderHandler extends BulkbuilderTestBase {

  /**
   * Test creation of an add bulk only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddBulk() throws Exception {
    _log.info("testIncModeAddBulkWithHandler start.");
    final String jobName = "testIncModeAddBulkWithHandler";
    final String jobId = startJob(jobName, "tempstore");
    try {
      final int noOfRecords = 5;
      for (int i = 0; i < noOfRecords; i++) {
        addRecordWithClient(jobName, createRecord(i));
      }
      commitRecordWithClient(jobName);
      assertNoTask(WORKER_DELETESCONSUMER);
      waitForFinishingTasksProcessed();
      final Task task = getTask(WORKER_BULKCONSUMER);
      final RecordInput reader = readBulk(task, "bulk");
      try {
        final Set<Integer> idSet = new HashSet<Integer>();
        for (int i = 0; i < noOfRecords; i++) {
          final Record record = reader.getRecord();
          assertNotNull(record);
          idSet.add(getRecordIdAsInteger(record));
        }
        assertEquals(noOfRecords, idSet.size());
        finishTaskWithSuccess(task);
      } finally {
        reader.close();
      }
      assertNoTask(WORKER_BULKCONSUMER);
    } finally {
      _jobRunEngine.finishJob(jobName, jobId);
      _log.info("testIncModeAddBulkWithHandler end.");
    }
  }

  /**
   * Test creation of a delete bulk only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testDeleteBulk() throws Exception {
    _log.info("testIncModeDeleteBulkWithHandler start.");
    final String jobName = "testIncModeDeleteBulkWithHandler";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 5;
    for (int i = 0; i < noOfRecords; i++) {
      deleteRecordWithClient(jobName, i);
    }
    commitRecordWithClient(jobName);
    assertNoTask(WORKER_BULKCONSUMER);
    waitForFinishingTasksProcessed();
    final Task task = getTask(WORKER_DELETESCONSUMER);
    final RecordInput reader = readBulk(task, "deletes");
    try {
      final Set<Integer> idSet = new HashSet<Integer>();
      for (int i = 0; i < noOfRecords; i++) {
        final Record record = reader.getRecord();
        assertNotNull(record);
        idSet.add(getRecordIdAsInteger(record));
      }
      assertEquals(noOfRecords, idSet.size());
      finishTaskWithSuccess(task);
      assertNoTask(WORKER_DELETESCONSUMER);
    } finally {
      reader.close();
    }
    _jobRunEngine.finishJob(jobName, jobId);
    _log.info("testDeleteBulkWithHandler end.");
  }

  /**
   * Test creation of an add and delete bulk.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddAndDeleteBulk() throws Exception {
    _log.info("testIncModeAddAndDeleteBulkWithHandler start.");
    final String jobName = "testIncModeAddAndDeleteBulkWithHandler";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 5;
    for (int i = 0; i < noOfRecords; i++) {
      addRecordWithClient(jobName, createRecord(i));
    }
    for (int i = 0; i < noOfRecords; i++) {
      deleteRecordWithClient(jobName, noOfRecords + i);
    }
    commitRecordWithClient(jobName);
    waitForFinishingTasksProcessed();
    final Task addTask = getTask(WORKER_BULKCONSUMER);
    final RecordInput addReader = readBulk(addTask, "bulk");
    try {
      final Set<Integer> idSetAdds = new HashSet<Integer>();
      for (int i = 0; i < noOfRecords; i++) {
        final Record record = addReader.getRecord();
        assertNotNull(record);
        idSetAdds.add(getRecordIdAsInteger(record));
      }
      assertEquals(noOfRecords, idSetAdds.size());
      finishTaskWithSuccess(addTask);
    } finally {
      addReader.close();
    }
    assertNoTask(WORKER_BULKCONSUMER);
    final Task deleteTask = getTask(WORKER_DELETESCONSUMER);
    final RecordInput deleteReader = readBulk(deleteTask, "deletes");
    try {
      final Set<Integer> idSetDeletes = new HashSet<Integer>();
      for (int i = 0; i < noOfRecords; i++) {
        final Record record = deleteReader.getRecord();
        assertNotNull(record);
        idSetDeletes.add(getRecordIdAsInteger(record));
      }
      assertEquals(noOfRecords, idSetDeletes.size());
      finishTaskWithSuccess(deleteTask);
    } finally {
      deleteReader.close();
    }
    assertNoTask(WORKER_DELETESCONSUMER);
    _jobRunEngine.finishJob(jobName, jobId);
    _log.info("testAddAndDeleteBulkWithHandler end.");
  }

  /**
   * Test creation and size-based commit of an add bulks only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddBulkBySize() throws Exception {
    _log.info("testIncModeAddBulkBySizeWithHandler start.");
    final String jobName = "testIncModeAddBulkBySizeWithHandler";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 50;
    final Set<Integer> idSet = new HashSet<Integer>();
    for (int i = 0; i < noOfRecords; i++) {
      addRecordWithClient(jobName, createRecord(i));
    }
    commitRecordWithClient(jobName);
    waitForFinishingTasksProcessed();
    int recordCount = 0;
    assertNoTask(WORKER_DELETESCONSUMER);
    Task task = null;
    do {
      task = _taskManager.getTask(WORKER_BULKCONSUMER, null);
      if (task != null) {
        final RecordInput reader = readBulk(task, "bulk");
        try {
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              recordCount++;
              idSet.add(getRecordIdAsInteger(record));
            }
          } while (record != null);
          finishTaskWithSuccess(task);
        } finally {
          reader.close();
        }
      }
    } while (task != null);
    assertEquals(noOfRecords, idSet.size());
    assertNoTask(WORKER_BULKCONSUMER);
    assertEquals(noOfRecords, recordCount);
    _jobRunEngine.finishJob(jobName, jobId);
    _log.info("testIncModeAddBulkBySizeWithHandler end.");
  }

  /**
   * Test creation and size-based commit of an delete bulks only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testDeleteBulkBySize() throws Exception {
    _log.info("testIncModeDeleteBulkBySizeWithHandler start.");
    final String jobName = "testIncModeDeleteBulkBySizeWithHandler";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 50;
    final Set<Integer> idSet = new HashSet<Integer>();
    for (int i = 0; i < noOfRecords; i++) {
      deleteBigRecordWithClient(jobName, i);
    }
    commitRecordWithClient(jobName);
    waitForFinishingTasksProcessed();
    int recordCount = 0;
    assertNoTask(WORKER_BULKCONSUMER);
    Task task = null;
    do {
      task = _taskManager.getTask(WORKER_DELETESCONSUMER, null);
      if (task != null) {
        final RecordInput reader = readBulk(task, "deletes");
        try {
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              recordCount++;
              idSet.add(getRecordIdAsInteger(record));
            }
          } while (record != null);
          finishTaskWithSuccess(task);
        } finally {
          reader.close();
        }
      }
    } while (task != null);
    assertEquals(noOfRecords, idSet.size());
    assertNoTask(WORKER_DELETESCONSUMER);
    assertEquals(noOfRecords, recordCount);
    _jobRunEngine.finishJob(jobName, jobId);
    _log.info("testIncModeDeleteBulkBySizeWithHandler end.");
  }

  /**
   * tests if the error return codes are correct.
   * 
   * @throws Exception
   *           unexpected Exception
   */
  public void testErrorCodes() throws Exception {
    // test post to job that does not exist
    HttpMethod method = new PostMethod(BASE_URI + "job/iDoNotExist/record");
    assertEquals(HttpStatus.SC_NOT_FOUND, _httpClient.executeMethod(method));

    final String jobName = "testIncModeErrorCodesWithHandler";
    createJob(jobName, "bulkBuilderTest", "test");

    // test post to job that does not run yet
    method = new PostMethod(BASE_URI + "job/" + jobName + "/record");
    assertEquals(HttpStatus.SC_NOT_FOUND, _httpClient.executeMethod(method));

    // test DELETE to job that does not run yet
    method = new DeleteMethod(BASE_URI + "job/" + jobName + "/record");
    assertEquals(HttpStatus.SC_NOT_FOUND, _httpClient.executeMethod(method));

    final String jobRunId = _jobRunEngine.startJob(jobName);

    // test DELETE with no record id set

    method = new DeleteMethod(BASE_URI + "job/" + jobName + "/record?content=test");
    assertEquals(HttpStatus.SC_BAD_REQUEST, _httpClient.executeMethod(method));

    // test ADD without no record id set
    try {
      addRecordWithClient(jobName, createRecordWithoutId());
    } catch (final HttpClientException e) {
      assertEquals(HttpStatus.SC_BAD_REQUEST, e.getResponseCode());
    }

    _jobRunEngine.finishJob(jobName, jobRunId);

    // test POST to job that has already been finished
    method = new PostMethod(BASE_URI + "job/" + jobName + "/record");
    assertEquals(HttpStatus.SC_NOT_FOUND, _httpClient.executeMethod(method));

    // test DELETE to job that has already been finished
    method = new DeleteMethod(BASE_URI + "job/" + jobName + "/record?_recordid=1");
    assertEquals(HttpStatus.SC_NOT_FOUND, _httpClient.executeMethod(method));

    // test put to add url
    method = new PutMethod(BASE_URI + "job/iDonotExist/record");
    assertEquals(HttpStatus.SC_METHOD_NOT_ALLOWED, _httpClient.executeMethod(method));

    // test put to del url#
    method = new PutMethod(BASE_URI + "job/iDoNotExist/record?_recordid=1");
    assertEquals(HttpStatus.SC_METHOD_NOT_ALLOWED, _httpClient.executeMethod(method));
  }

  /** test adding a record with one attachments. */
  public void testAddRecordsWithOneAttachment() throws Exception {
    _log.info("testAddRecordsWithOneAttachment start.");
    final String jobName = "testAddRecordsWithOneAttachment";
    final String jobId = startJob(jobName, "tempstore");
    try {
      final int noOfRecords = 1;
      final File attachment = ConfigUtils.getConfigFile("org.eclipse.smila.bulkbuilder", "smila-icon.icns");
      final byte[] expectedAttachmentContent = FileUtils.readFileToByteArray(attachment);
      final Map<String, File> attachments = new HashMap<String, File>();
      final String expectedAttachmentName = "file";
      attachments.put(expectedAttachmentName, attachment);
      for (int i = 0; i < noOfRecords; i++) {
        addRecordAndAttachFiles(jobName, createRecord(i), attachments, i % 2 == 0);
      }
      commitRecordWithClient(jobName);
      assertNoTask(WORKER_DELETESCONSUMER);
      waitForFinishingTasksProcessed();
      final Task task = getTask(WORKER_BULKCONSUMER);
      final RecordInput reader = readBulk(task, "bulk");
      try {
        final Set<Integer> idSet = new HashSet<Integer>();
        for (int i = 0; i < noOfRecords; i++) {
          final Record record = reader.getRecord();
          assertNotNull(record);
          idSet.add(getRecordIdAsInteger(record));
          assertEquals(1, record.attachmentSize());
          final Iterator<String> attachmentNames = record.getAttachmentNames();
          assertTrue(attachmentNames.hasNext());
          assertEquals(expectedAttachmentName, attachmentNames.next());
          assertTrue(Arrays.equals(expectedAttachmentContent, record.getAttachmentAsBytes(expectedAttachmentName)));
          assertFalse(attachmentNames.hasNext());
        }
        assertEquals(noOfRecords, idSet.size());
        finishTaskWithSuccess(task);
      } finally {
        reader.close();
      }
      assertNoTask(WORKER_BULKCONSUMER);
    } finally {
      _jobRunEngine.finishJob(jobName, jobId);
      _log.info("testAddRecordsWithOneAttachment end.");
    }
  }

  /** test adding a record with multiple attachments. */
  public void testAddRecordsWithAttachments() throws Exception {
    _log.info("testAddRecordsWithOneAttachment start.");
    final String jobName = "testAddRecordsWithAttachments";
    final String jobId = startJob(jobName, "tempstore");
    try {
      final int noOfRecords = 5;
      final File[] attachmentFiles =
        new File[] { ConfigUtils.getConfigFile("org.eclipse.smila.jobmanager", "buckets.json"),
          ConfigUtils.getConfigFile("org.eclipse.smila.jobmanager", "dataObjectTypes.json"),
          ConfigUtils.getConfigFile("org.eclipse.smila.jobmanager", "workers.json"),
          ConfigUtils.getConfigFile("org.eclipse.smila.jobmanager", "workflows.json") };
      final byte[][] expectedAttachmentContents = new byte[4][];
      final Map<String, File> attachments = new LinkedHashMap<String, File>();
      for (int j = 0; j < attachmentFiles.length; j++) {
        expectedAttachmentContents[j] = FileUtils.readFileToByteArray(attachmentFiles[j]);
        attachments.put("file" + j, attachmentFiles[j]);
      }
      for (int i = 0; i < noOfRecords; i++) {
        addRecordAndAttachFiles(jobName, createRecord(i), attachments, i % 2 == 0);
      }
      commitRecordWithClient(jobName);
      assertNoTask(WORKER_DELETESCONSUMER);
      waitForFinishingTasksProcessed();

      Task task = null;
      final Set<Integer> idSet = new HashSet<Integer>();
      do {
        task = _taskManager.getTask(WORKER_BULKCONSUMER, null);
        if (task != null) {
          final RecordInput reader = readBulk(task, "bulk");
          try {
            Record record = null;
            do {
              record = reader.getRecord();
              if (record != null) {
                idSet.add(getRecordIdAsInteger(record));
                assertEquals(attachmentFiles.length, record.attachmentSize());
                final Iterator<String> attachmentNames = record.getAttachmentNames();
                for (int j = 0; j < attachmentFiles.length; j++) {
                  assertTrue("no more attachments at j=" + j, attachmentNames.hasNext());
                  final String expectedAttachmentName = "file" + j;
                  assertEquals(expectedAttachmentName, attachmentNames.next());
                  assertTrue(Arrays.equals(expectedAttachmentContents[j],
                    record.getAttachmentAsBytes(expectedAttachmentName)));
                }
              }
            } while (record != null);
          } finally {
            reader.close();
            finishTaskWithSuccess(task);
          }
        }
      } while (task != null);
      assertEquals(noOfRecords, idSet.size());
      assertNoTask(WORKER_BULKCONSUMER);
    } finally {
      _jobRunEngine.finishJob(jobName, jobId);
      _log.info("testAddRecordsWithOneAttachment end.");
    }
  }

  /** test adding a record with an attachments that is too large attachments. */
  public void testFailAddRecordWithExcessiveAttachment() throws Exception {
    _log.info("testFailAddRecordWithExcessiveAttachment start.");
    final String jobName = "testFailAddRecordWithExcessiveAttachment";
    final String jobId = startJob(jobName, "tempstore");
    try {
      final File workspaceDir = WorkspaceHelper.createWorkingDir(AllTests.BUNDLE_ID);
      final File tooBigAttachment = new File(workspaceDir, "too-big-data");
      // configured max. is 100k, see httpserver.properties in configuration directory.
      FileUtils.writeByteArrayToFile(tooBigAttachment, new byte[200000]);
      final Map<String, File> attachments = new HashMap<String, File>();
      attachments.put("tooBig", tooBigAttachment);
      addRecordAndAttachFiles(jobName, createRecord(0), attachments, false);
      fail("should not work");
    } catch (final HttpClientException ex) {
      assertEquals(HttpStatus.SC_BAD_REQUEST, ex.getResponseCode());
    } finally {
      _jobRunEngine.finishJob(jobName, jobId);
      _log.info("testFailAddRecordWithExcessiveAttachment end.");
    }
  }

  /** test adding a record with an attachments that is too large attachments. */
  public void testFailAddRecordWithExcessiveAttachmentChunked() throws Exception {
    _log.info("testFailAddRecordWithExcessiveAttachment start.");
    final String jobName = "testAddRecordWithExcessiveAttachment";
    final String jobId = startJob(jobName, "tempstore");
    try {
      final File workspaceDir = WorkspaceHelper.createWorkingDir(AllTests.BUNDLE_ID);
      final File tooBigAttachment = new File(workspaceDir, "too-big-data");
      // configured max. is 100k, see httpserver.properties in configuration directory.
      FileUtils.writeByteArrayToFile(tooBigAttachment, new byte[200000]);
      final Map<String, File> attachments = new HashMap<String, File>();
      attachments.put("tooBig", tooBigAttachment);
      addRecordAndAttachFiles(jobName, createRecord(0), attachments, true);
      fail("should not work");
    } catch (final HttpClientException ex) {
      assertEquals(HttpStatus.SC_BAD_REQUEST, ex.getResponseCode());
    } finally {
      _jobRunEngine.finishJob(jobName, jobId);
      _log.info("testFailAddRecordWithExcessiveAttachment end.");
    }
  }

  /** test adding a record with an attachments that is too large attachments. */
  public void testFailAddRecordWithTwoAttachmentsExceedingLimitChunked() throws Exception {
    _log.info("testFailAddRecordWithTwoAttachmentsExceedingLimitChunked start.");
    final String jobName = "testFailAddRecordWithTwoAttachmentsExceedingLimitChunked";
    final String jobId = startJob(jobName, "tempstore");
    try {
      final File attachment = ConfigUtils.getConfigFile("org.eclipse.smila.bulkbuilder", "smila-icon.icns");
      // configured max. is 100k, see httpserver.properties in configuration directory.
      final Map<String, File> attachments = new HashMap<String, File>();
      attachments.put("oneIsOk", attachment);
      attachments.put("twoIsTooBig", attachment);
      addRecordAndAttachFiles(jobName, createRecord(0), attachments, true);
      fail("should not work");
    } catch (final HttpClientException ex) {
      assertEquals(HttpStatus.SC_BAD_REQUEST, ex.getResponseCode());
    } finally {
      _jobRunEngine.finishJob(jobName, jobId);
      _log.info("testFailAddRecordWithTwoAttachmentsExceedingLimitChunked end.");
    }
  }

  /** test adding a record with an attachments that is too large attachments. */
  public void testFailAddRecordWithAttachmentsJsonExceedingLimitChunked() throws Exception {
    _log.info("testFailAddRecordWithAttachmentsJsonExceedingLimitChunked start.");
    final String jobName = "testFailAddRecordWithAttachmentsJsonExceedingLimitChunked";
    final String jobId = startJob(jobName, "tempstore");
    try {
      final Record record = createRecord(0);
      final char[] value = new char[200000];
      Arrays.fill(value, 'x');
      record.getMetadata().put("bigAttribute", new String(value));
      final File attachment = ConfigUtils.getConfigFile("org.eclipse.smila.bulkbuilder", "bulkbuilder.properties");
      final Map<String, File> attachments = new HashMap<String, File>();
      attachments.put("smallAttachment", attachment);
      addRecordAndAttachFiles(jobName, record, attachments, true);
      fail("should not work");
    } catch (final HttpClientException ex) {
      assertEquals(HttpStatus.SC_BAD_REQUEST, ex.getResponseCode());
    } finally {
      _jobRunEngine.finishJob(jobName, jobId);
      _log.info("testFailAddRecordWithAttachmentsJsonExceedingLimitChunked end.");
    }
  }

  /** test adding a record with an attachments that is too large attachments. */
  public void testFailAddRecordWithJsonExceedingLimitChunked() throws Exception {
    _log.info("testFailAddRecordWithJsonExceedingLimitChunked start.");
    final String jobName = "testFailAddRecordWithJsonExceedingLimitChunked";
    final String jobId = startJob(jobName, "tempstore");
    try {
      final Record record = createRecord(0);
      final char[] value = new char[200000];
      Arrays.fill(value, 'x');
      record.getMetadata().put("bigAttribute", new String(value));
      addRecordWithClient(jobName, record, true);
      fail("should not work");
    } catch (final HttpClientException ex) {
      assertEquals(HttpStatus.SC_BAD_REQUEST, ex.getResponseCode());
    } finally {
      _jobRunEngine.finishJob(jobName, jobId);
      _log.info("testFailAddRecordWithJsonExceedingLimitChunked end.");
    }
  }

}
