/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Drazen Cindric (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.test;

import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.ValueFormatHelper;

/**
 * Test class for testing the methods of Value and ValueImpl.
 * 
 * @author cind01
 * 
 */
public class TestValue extends TestCase {

  /**
   * Factory for creating Any objects.
   */
  private final DataFactory _dataFactory = DataFactory.DEFAULT;

  /**
   * Test creation of a String value and respective methods.
   * 
   * @throws Exception
   *           An exception if something goes wrong.
   */
  public void testCreateStringValue() throws Exception {
    // test creation and correct type
    final String testString = "This is a test!";
    Value value = _dataFactory.createStringValue(testString);
    assertNotNull(value);
    assertTrue(value.isString());
    assertEquals(value.getValueType(), Any.ValueType.STRING);
    assertFalse(value.isDouble());
    assertFalse(value.isBoolean());
    assertFalse(value.isDate());
    assertFalse(value.isDateTime());
    assertFalse(value.isLong());
    Double doubleValue = null;
    try {
      doubleValue = value.asDouble();
      fail("Should not happen, an InvalidValueTypeException should occur.");
    } catch (final InvalidValueTypeException ivte) {
      assertNotNull(ivte);
      assertNull(doubleValue);
    }

    // test equals and hash code
    final Value stringValue = _dataFactory.createStringValue(testString);
    assertTrue(value.equals(stringValue));
    assertEquals(value.hashCode(), stringValue.hashCode());
    assertFalse(value.equals(null));

    // test double string value
    value = _dataFactory.createStringValue("5.0");
    assertNotNull(value);
    assertTrue(value.isString());
    assertFalse(value.isDouble());
    assertTrue(value.asDouble() instanceof Double);

    // test long string value
    value = _dataFactory.createStringValue("5");
    assertNotNull(value);
    assertTrue(value.isString());
    assertFalse(value.isDouble());
    assertTrue(value.asLong() instanceof Long);

    // test boolean string value
    value = _dataFactory.createStringValue("true");
    assertNotNull(value);
    assertTrue(value.isString());
    assertFalse(value.isDouble());
    assertTrue(value.asBoolean());

    // test date string value
    value = _dataFactory.createStringValue("1973-05-29");
    assertNotNull(value);
    assertTrue(value.isString());
    assertFalse(value.isDouble());
    assertTrue(value.asDate() instanceof Date);

    // test datetime string value
    value = _dataFactory.createStringValue("1973-05-29T07:30:00+0000");
    assertNotNull(value);
    assertTrue(value.isString());
    assertFalse(value.isDouble());
    assertTrue(value.asDateTime() instanceof Date);
  }

  /**
   * Test creation of a Double value and respective methods.
   * 
   * @throws Exception
   *           An exception if something goes wrong.
   */
  public void testCreateDouble() throws Exception {
    // test creation and correct type
    final int testNumber = 12345;
    final Double testDouble = new Double(testNumber);
    final Value value = _dataFactory.createDoubleValue(testDouble);
    assertNotNull(value);
    assertTrue(value.isDouble());
    assertTrue(value.isNumber());
    assertEquals(value.getValueType(), Any.ValueType.DOUBLE);
    assertFalse(value.isString());
    assertFalse(value.isBoolean());
    assertFalse(value.isDate());
    assertFalse(value.isDateTime());
    assertFalse(value.isLong());

    Long longValue = value.asLong(); // should work
    assertEquals(new Long(testNumber), longValue);

    try {
      value.asBoolean();
      fail("Should not happen, an InvalidValueTypeException should occur.");
    } catch (final InvalidValueTypeException ivte) {
      assertNotNull(ivte);
    }

    // test equals and hash code
    final Value doubleValue = _dataFactory.createDoubleValue(testDouble);
    assertTrue(value.equals(doubleValue));
    assertEquals(value.hashCode(), doubleValue.hashCode());
    assertFalse(value.equals(null));

    // test for different type even if their internal value 12345 seems to be equal
    longValue = new Long(testNumber);
    assertFalse(value.equals(_dataFactory.createLongValue(longValue)));
  }

  /**
   * Test creation of a Long value and respective methods.
   * 
   * @throws Exception
   *           An exception if something goes wrong.
   */
  public void testCreateLong() throws Exception {
    // test creation and correct type
    final int testNumber = 12345;
    final Long testLong = new Long(testNumber);
    final Value value = _dataFactory.createLongValue(testLong);
    assertNotNull(value);
    assertTrue(value.isLong());
    assertTrue(value.isNumber());
    assertFalse(value.isDouble());
    assertEquals(value.getValueType(), Any.ValueType.LONG);
    assertFalse(value.isString());
    assertFalse(value.isBoolean());
    assertFalse(value.isDate());
    assertFalse(value.isDateTime());

    Double doubleValue = value.asDouble();
    assertEquals(testLong, new Long(doubleValue.longValue()));

    try {
      value.asDate();
      fail("Should not happen, an InvalidValueTypeException should occur.");
    } catch (final InvalidValueTypeException ivte) {
      assertNotNull(ivte);
    }

    // test equals and hash code
    final Value longValue = _dataFactory.createLongValue(testLong);
    assertTrue(value.equals(longValue));
    assertEquals(value.hashCode(), longValue.hashCode());
    assertFalse(value.equals(null));

    // test for different type even if their internal value 12345 seems to be equal
    doubleValue = new Double(testNumber);
    assertFalse(value.equals(_dataFactory.createDoubleValue(doubleValue)));
  }

  /**
   * Test creation of a Boolean value and respective methods.
   * 
   * @throws Exception
   *           An exception if something goes wrong.
   */
  public void testCreateBoolean() throws Exception {
    // test creation and correct type
    final String testString = "true";
    final Boolean testBoolean = Boolean.parseBoolean(testString);
    final Value value = _dataFactory.createBooleanValue(testBoolean);
    assertNotNull(value);
    assertTrue(value.isBoolean());
    assertFalse(value.isNumber());
    assertEquals(value.getValueType(), Any.ValueType.BOOLEAN);
    assertFalse(value.isDouble());
    assertFalse(value.isString());
    assertFalse(value.isDate());
    assertFalse(value.isDateTime());
    assertFalse(value.isLong());
    String stringValue = null;

    // /here no exception should occur
    stringValue = value.asString();
    assertEquals(testString, stringValue);

    // test equals and hash code
    final Value booleanValue = _dataFactory.createBooleanValue(testBoolean);
    assertTrue(value.equals(booleanValue));
    assertEquals(value.hashCode(), booleanValue.hashCode());
    assertFalse(value.equals(null));

    // test for different type even if their internal value 12345 seems to be equal
    assertFalse(value.equals(_dataFactory.createStringValue(testString)));
  }

  /**
   * Test creation of a date value and respective methods.
   * 
   * @throws Exception
   *           An exception if something goes wrong.
   */
  public void testCreateDate() throws Exception {
    // test creation and correct type
    final Calendar cal = Calendar.getInstance();
    cal.set(Calendar.HOUR_OF_DAY, 0);
    cal.set(Calendar.MINUTE, 0);
    cal.set(Calendar.SECOND, 0);
    cal.set(Calendar.MILLISECOND, 0);
    final Date testDate = cal.getTime();

    final Value value = _dataFactory.createDateValue(testDate);
    assertNotNull(value);
    assertTrue(value.isDate());
    assertFalse(value.isNumber());
    assertFalse(value.isDouble());
    assertEquals(value.getValueType(), Any.ValueType.DATE);
    assertFalse(value.isString());
    assertFalse(value.isBoolean());
    assertFalse(value.isLong());
    assertFalse(value.isDateTime());
    assertEquals(value.asDateTime(), value.asDate());

    try {
      value.asLong();
      fail("Should not happen, an InvalidValueTypeException should occur.");
    } catch (final InvalidValueTypeException ivte) {
      assertNotNull(ivte);
    }

    // test equals and hash code
    final Value dateValue = _dataFactory.createDateValue(testDate);
    assertTrue(value.equals(dateValue));
    assertEquals(value.hashCode(), dateValue.hashCode());
    assertFalse(value.equals(null));

    final Date parsedDate = ValueFormatHelper.INSTANCE.parseDate(value.asString());
    assertEquals(testDate, parsedDate);
  }

  /**
   * Test creation of a datetime value and respective methods.
   * 
   * @throws Exception
   *           An exception if something goes wrong.
   */
  public void testCreateDateTime() throws Exception {
    final Calendar cal = Calendar.getInstance();
    cal.set(Calendar.MILLISECOND, 0);
    final Date testDate = cal.getTime();
    final Value value = _dataFactory.createDateTimeValue(testDate);
    assertNotNull(value);
    assertTrue(value.isDateTime());
    assertFalse(value.isNumber());
    assertFalse(value.isDouble());
    assertEquals(value.getValueType(), Any.ValueType.DATETIME);
    assertFalse(value.isString());
    assertFalse(value.isBoolean());
    assertFalse(value.isLong());
    assertFalse(value.isDate());
    assertEquals(value.asDateTime(), value.asDate());

    // test for different type even if their internal value the testDate seems to be equal
    try {
      value.asDouble();
      fail("Should not happen, an InvalidValueTypeException should occur.");
    } catch (final InvalidValueTypeException ivte) {
      assertNotNull(ivte);
    }

    // test equals and hash code
    final Value dateTimeValue = _dataFactory.createDateTimeValue(testDate);
    assertTrue(value.equals(dateTimeValue));
    assertEquals(value.hashCode(), dateTimeValue.hashCode());
    assertFalse(value.equals(null));

    final Date parsedDate = ValueFormatHelper.INSTANCE.parseDateTime(value.asString());
    assertEquals(testDate.getTime() / 1000, parsedDate.getTime() / 1000);
  }

  /** test parsing of date values from string. */
  public void testParseDate() throws Exception {
    final Date date1 = ValueFormatHelper.INSTANCE.parseDate("2001-02-03");
    final Calendar calendar = Calendar.getInstance();
    calendar.setTime(date1);
    assertEquals(2001, calendar.get(Calendar.YEAR));
    assertEquals(Calendar.FEBRUARY, calendar.get(Calendar.MONTH));
    assertEquals(3, calendar.get(Calendar.DAY_OF_MONTH));
    final Date date2 = ValueFormatHelper.INSTANCE.parseDate("0050-02-03");
    calendar.setTime(date2);
    assertEquals(50, calendar.get(Calendar.YEAR));
    assertEquals(Calendar.FEBRUARY, calendar.get(Calendar.MONTH));
    assertEquals(3, calendar.get(Calendar.DAY_OF_MONTH));
    try {
      ValueFormatHelper.INSTANCE.parseDate("2001-02-031");
      fail("should not work");
    } catch (final ParseException ex) {
      ; // OK
    }
  }

  /** test parsing of datetime values from string. */
  public void testParseDateTime() throws Exception {
    // test datetime pattern 1
    final Date date1 = ValueFormatHelper.INSTANCE.parseDateTime("2001-02-03T12:34:56.123+0000");
    final Calendar calendar = Calendar.getInstance();
    final long zoneOffset = calendar.get(Calendar.ZONE_OFFSET);
    final long hourOffset = zoneOffset / 3600 / 1000;
    calendar.setTime(date1);
    assertEquals(2001, calendar.get(Calendar.YEAR));
    assertEquals(Calendar.FEBRUARY, calendar.get(Calendar.MONTH));
    assertEquals(3, calendar.get(Calendar.DAY_OF_MONTH));
    final int hour1 = calendar.get(Calendar.HOUR_OF_DAY);
    assertEquals(12 + hourOffset, hour1);
    assertEquals(34, calendar.get(Calendar.MINUTE));
    assertEquals(56, calendar.get(Calendar.SECOND));
    assertEquals(123, calendar.get(Calendar.MILLISECOND));
    assertEquals(zoneOffset, calendar.get(Calendar.ZONE_OFFSET));

    // test datetime pattern 2
    final Date date2 = ValueFormatHelper.INSTANCE.parseDateTime("2001-02-03T12:34:56+0100");
    calendar.setTime(date2);
    assertEquals(2001, calendar.get(Calendar.YEAR));
    assertEquals(Calendar.FEBRUARY, calendar.get(Calendar.MONTH));
    assertEquals(3, calendar.get(Calendar.DAY_OF_MONTH));
    final int hour2 = calendar.get(Calendar.HOUR_OF_DAY);
    assertEquals(11 + hourOffset, hour2);
    assertEquals(hour1 - 1, hour2);
    assertEquals(34, calendar.get(Calendar.MINUTE));
    assertEquals(56, calendar.get(Calendar.SECOND));
    assertEquals(zoneOffset, calendar.get(Calendar.ZONE_OFFSET));

    // test datetime pattern 1
    final Date date3 = ValueFormatHelper.INSTANCE.parseDateTime("2001-02-03T12:34:56-0200");
    calendar.setTime(date3);
    assertEquals(2001, calendar.get(Calendar.YEAR));
    assertEquals(Calendar.FEBRUARY, calendar.get(Calendar.MONTH));
    assertEquals(3, calendar.get(Calendar.DAY_OF_MONTH));
    final int hour3 = calendar.get(Calendar.HOUR_OF_DAY);
    assertEquals(14 + hourOffset, hour3);
    assertEquals(hour1 + 2, hour3);
    assertEquals(34, calendar.get(Calendar.MINUTE));
    assertEquals(56, calendar.get(Calendar.SECOND));
    assertEquals(zoneOffset, calendar.get(Calendar.ZONE_OFFSET));

    // test parse exception
    try {
      ValueFormatHelper.INSTANCE.parseDateTime("2001-02-03T12:34:56+00001");
      fail("should not work");
    } catch (final ParseException ex) {
      ; // OK
    }

    // test datetime pattern 3
    final Date date4 = ValueFormatHelper.INSTANCE.parseDateTime("2001-02-03T12:34:56.123Z");
    calendar.setTime(date4);
    assertEquals(2001, calendar.get(Calendar.YEAR));
    assertEquals(Calendar.FEBRUARY, calendar.get(Calendar.MONTH));
    assertEquals(3, calendar.get(Calendar.DAY_OF_MONTH));
    final int hour4 = calendar.get(Calendar.HOUR_OF_DAY);
    assertEquals(12 + hourOffset, hour4);
    assertEquals(34, calendar.get(Calendar.MINUTE));
    assertEquals(56, calendar.get(Calendar.SECOND));
    assertEquals(123, calendar.get(Calendar.MILLISECOND));
    assertEquals(zoneOffset, calendar.get(Calendar.ZONE_OFFSET));

    // test datetime pattern 4
    final Date date5 = ValueFormatHelper.INSTANCE.parseDateTime("2001-02-03T12:34:56Z");
    calendar.setTime(date5);
    assertEquals(2001, calendar.get(Calendar.YEAR));
    assertEquals(Calendar.FEBRUARY, calendar.get(Calendar.MONTH));
    assertEquals(3, calendar.get(Calendar.DAY_OF_MONTH));
    final int hour5 = calendar.get(Calendar.HOUR_OF_DAY);
    assertEquals(12 + hourOffset, hour5);
    assertEquals(34, calendar.get(Calendar.MINUTE));
    assertEquals(56, calendar.get(Calendar.SECOND));
    assertEquals(zoneOffset, calendar.get(Calendar.ZONE_OFFSET));
  }

  /** test if a given original string is returned for date objects. */
  public void testOriginalDateString() {
    final Calendar nowCal = Calendar.getInstance();    
    nowCal.set(2000, 10, 10, 0, 0, 0);
    nowCal.set(Calendar.MILLISECOND, 0);
    final Date now = nowCal.getTime();    
    final String nowAsString = _dataFactory.createDateValue(now).asString();
    final Value value = _dataFactory.parseFromString(nowAsString, ValueType.DATE);
    System.out.println(now.getTime() - value.asDate().getTime());
    assertEquals(now, value.asDate());
    assertEquals(nowAsString, value.asString());
  }

  /** test if a given original string is returned for datetime objects. */
  public void testOriginalDateTimeString() {
    final Date now = new Date();
    final String nowAsString = _dataFactory.createDateTimeValue(now).asString();
    final Value value = _dataFactory.parseFromString(nowAsString, ValueType.DATETIME);    
    assertEquals(now, value.asDateTime());
    assertEquals(nowAsString, value.asString());
  }

  /**
   * Check how Value.toString with null.
   */
  public void testNullValue() {
    Value value = null;
    try {
      value = _dataFactory.createStringValue(null);
      fail("Should not happen, an IllegalArgumentException should occur.");
    } catch (final IllegalArgumentException iae) {
      assertNotNull(iae);
      assertNull(value);
    }
  }

  /**
   * test iterable methods of value.
   */
  public void testIterable() {
    final Value value = _dataFactory.createStringValue("iterable");
    assertFalse(value.isEmpty());
    assertEquals(1, value.size());
    final Iterator<Any> iterator = value.iterator();
    assertTrue(iterator.hasNext());
    assertSame(iterator.next(), value);
    assertFalse(iterator.hasNext());
    for (final Any element : value) {
      assertSame(element, value);
    }
  }
}
