/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.datamodel.util.test;

import java.util.ArrayList;
import java.util.Arrays;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.util.RecordMerger;

/**
 * Test cases for {@link RecordMerger}.
 */
public class TestRecordMerger extends TestCase {

  /** test record merging with overwriting existing attributes. */
  public void testOverwriteAttributes() {
    final Record r1 = DataFactory.DEFAULT.createRecord("r1");
    r1.getMetadata().put("attrib1", "v1");
    r1.getMetadata().put("attrib2", "v2");
    final AnyMap map1 = DataFactory.DEFAULT.createAnyMap();
    map1.put("key1", "value1");
    r1.getMetadata().put("attribMap", map1);
    r1.setAttachment("attach1", new byte[0]);

    final Record r2 = DataFactory.DEFAULT.createRecord("r2");
    r2.getMetadata().put("attrib2", "v2_new");
    r2.getMetadata().put("attrib3", "v3");
    final AnyMap map1_new = DataFactory.DEFAULT.createAnyMap();
    map1_new.put("key2", "value2");
    r2.getMetadata().put("attribMap", map1_new);
    r2.setAttachment("attach2", new byte[0]);

    RecordMerger.mergeRecords(r1, r2, false, false); // false -> overwrite attributes

    // check record r1 (to which r2 is merged)
    assertEquals("r1", r1.getId()); // record id is kept!
    assertEquals("v1", r1.getMetadata().get("attrib1").toString());
    assertEquals("v2_new", r1.getMetadata().get("attrib2").toString());
    assertEquals("v3", r1.getMetadata().get("attrib3").toString());
    assertEquals(Arrays.asList("key2"), new ArrayList<String>(r1.getMetadata().getMap("attribMap").keySet()));
    assertTrue(r1.hasAttachment("attach1"));
    assertTrue(r1.hasAttachment("attach2"));

    // check record r2 which should be unchanged
    assertEquals("r2", r2.getId());
    assertFalse(r2.getMetadata().containsKey("attrib1"));
    assertEquals("v2_new", r2.getMetadata().get("attrib2").toString());
    assertEquals("v3", r2.getMetadata().get("attrib3").toString());
    assertEquals(Arrays.asList("key2"), new ArrayList<String>(r1.getMetadata().getMap("attribMap").keySet()));
    assertFalse(r2.hasAttachment("attach1"));
    assertTrue(r2.hasAttachment("attach2"));
  }

  /** test record merging with merging existing attributes. */
  public void testMergeAttributes() {
    final Record r1 = DataFactory.DEFAULT.createRecord("r1");
    r1.getMetadata().put("attrib1", "v1");
    r1.getMetadata().put("attrib2", "v2");
    final AnySeq v3 = DataFactory.DEFAULT.createAnySeq();
    v3.add("v3_a");
    v3.add("v3_b");
    r1.getMetadata().put("attrib3", v3);
    r1.getMetadata().put("attrib4", "v4");
    final AnyMap map1 = DataFactory.DEFAULT.createAnyMap();
    map1.put("key1", "value1");
    r1.getMetadata().put("attribMap", map1);
    r1.setAttachment("attach1", "attachment".getBytes());

    final Record r2 = DataFactory.DEFAULT.createRecord("r2");
    r2.getMetadata().put("attrib2", "v2_new");
    final AnySeq v3_new = DataFactory.DEFAULT.createAnySeq();
    v3_new.add("v3_c");
    v3_new.add("v3_d");
    r2.getMetadata().put("attrib3", v3_new);
    final AnySeq v4_new = DataFactory.DEFAULT.createAnySeq();
    v4_new.add("v4_a");
    v4_new.add("v4_b");
    final AnyMap map1_new = DataFactory.DEFAULT.createAnyMap();
    map1_new.put("key2", "value2");
    r2.getMetadata().put("attribMap", map1_new);
    r2.getMetadata().put("attrib4", v4_new);
    r2.setAttachment("attach1", "attachment_new".getBytes());

    RecordMerger.mergeRecords(r1, r2, true, false); // true -> merge attributes

    // check record r1 (to which r2 is merged)
    assertEquals("r1", r1.getId()); // record id is kept!
    assertEquals("v1", r1.getMetadata().get("attrib1").toString());
    final AnySeq v2_merged = DataFactory.DEFAULT.createAnySeq();
    v2_merged.add("v2");
    v2_merged.add("v2_new");
    assertEquals(v2_merged, r1.getMetadata().get("attrib2"));
    final AnySeq v3_merged = DataFactory.DEFAULT.createAnySeq();
    v3_merged.add("v3_a");
    v3_merged.add("v3_b");
    v3_merged.add("v3_c");
    v3_merged.add("v3_d");
    assertEquals(v3_merged, r1.getMetadata().get("attrib3"));
    final AnySeq v4_merged = DataFactory.DEFAULT.createAnySeq();
    v4_merged.add("v4");
    v4_merged.add("v4_a");
    v4_merged.add("v4_b");
    assertEquals(v3_merged, r1.getMetadata().get("attrib3"));
    assertEquals(Arrays.asList("key1", "key2"),
      new ArrayList<String>(r1.getMetadata().getMap("attribMap").keySet()));
    // attachments will always be overwritten:
    assertEquals("attachment_new", new String(r1.getAttachmentAsBytes("attach1")));

    // check record r2 which should be unchanged
    assertEquals("r2", r2.getId());
    assertFalse(r2.getMetadata().containsKey("attrib1"));
    assertEquals("v2_new", r2.getMetadata().get("attrib2").toString());
    assertEquals(v3_new, r2.getMetadata().get("attrib3"));
    assertEquals("attachment_new", new String(r1.getAttachmentAsBytes("attach1")));
  }

  /** see issue @link https://bugs.eclipse.org/bugs/show_bug.cgi?id=403828 */
  public void testMergeWithValuesNotContainedInFirstRecord() throws Exception {
    final Record r1 = DataFactory.DEFAULT.createRecord("r1");
    r1.getMetadata().put("attrib1", "v1");
    final Record r2 = DataFactory.DEFAULT.createRecord("r2");
    r2.getMetadata().put("attrib2", "v2");

    final Record r3 = DataFactory.DEFAULT.createRecord("r3");
    r3.getMetadata().put("attrib3", "v3"); // this should be contained in the merge result!

    RecordMerger.mergeRecords(r1, r3, true, false);
    assertEquals("v1", r1.getMetadata().get("attrib1").toString());
    assertEquals("v3", r1.getMetadata().get("attrib3").toString());

    RecordMerger.mergeRecords(r2, r3, false, false);
    assertEquals("v2", r2.getMetadata().get("attrib2").toString());
    assertEquals("v3", r2.getMetadata().get("attrib3").toString());
  }

  /** test attachment overwrite. */
  public void testOverwriteAttachments() {
    final Record r1 = DataFactory.DEFAULT.createRecord("r1");
    r1.setAttachment("attach1", new byte[] { 8 });

    final Record r2 = DataFactory.DEFAULT.createRecord("r2");
    r2.setAttachment("attach1", new byte[] { 16 });
    r2.setAttachment("attach2", new byte[] { 32 });

    // overwrite attachments
    RecordMerger.mergeRecords(r1, r2, false, false);

    assertTrue(r1.hasAttachment("attach1"));
    assertEquals(1, r1.getAttachmentAsBytes("attach1").length);
    assertEquals(16, r1.getAttachmentAsBytes("attach1")[0]);

    assertTrue(r1.hasAttachment("attach2"));
    assertEquals(1, r1.getAttachmentAsBytes("attach2").length);
    assertEquals(32, r1.getAttachmentAsBytes("attach2")[0]);
  }

  /** test attachment merging. */
  public void testMergeAttachments() {
    final Record r1 = DataFactory.DEFAULT.createRecord("r1");
    r1.setAttachment("attach1", new byte[] { 8 });

    final Record r2 = DataFactory.DEFAULT.createRecord("r2");
    r2.setAttachment("attach1", new byte[] { 16 });
    r2.setAttachment("attach2", new byte[] { 32 });

    // merge attachments
    RecordMerger.mergeRecords(r1, r2, false, true);

    assertTrue(r1.hasAttachment("attach1"));
    assertEquals(2, r1.getAttachmentAsBytes("attach1").length);
    assertEquals(8, r1.getAttachmentAsBytes("attach1")[0]);
    assertEquals(16, r1.getAttachmentAsBytes("attach1")[1]);

    assertTrue(r1.hasAttachment("attach2"));
    assertEquals(1, r1.getAttachmentAsBytes("attach2").length);
    assertEquals(32, r1.getAttachmentAsBytes("attach2")[0]);
  }

}
