/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.test;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.http.client.BulkResponse;
import org.eclipse.smila.http.client.HttpMethod;
import org.eclipse.smila.http.client.RestClient;
import org.eclipse.smila.http.client.RestException;
import org.eclipse.smila.http.client.attachments.AttachmentWrapper;
import org.eclipse.smila.http.client.impl.DefaultRestClient;
import org.eclipse.smila.http.client.impl.base.HttpRequestFactory;
import org.eclipse.smila.http.server.HttpService;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/** unit tests for the SMILA REST client. */
public class TestDefaultRestClient extends DeclarativeServiceTestCase {

  /** param to add for getting a null response. */
  protected static final String NORESPONSE_PARAM = "?noResponse=1";

  /** the URI of the mock handler. */
  protected String _mockPath = "/smila/mock";

  /** the URI of the any handler. */
  protected String _anyPath = "/smila/any";

  /** the client to test. */
  protected RestClient _client = new DefaultRestClient();

  @Override
  protected void setUp() throws Exception {
    getService(HttpService.class);
  }

  @Override
  protected void tearDown() throws Exception {
    _client.shutdown();
  }

  /** test a simple get without parameters. */
  public void testGet() throws Exception {
    final AnyMap response = _client.get(_mockPath);
    assertNotNull(response);
    assertEquals(2, response.size());
    assertEquals("GET", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
  }

  /** test a get with parameters. */
  public void testGetWithParameters() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    parameters.put("long", 42);
    parameters.put("double", Math.PI);
    parameters.put("boolean", true);
    final AnyMap response = _client.get(_mockPath, parameters);
    assertNotNull(response);
    assertEquals(6, response.size());
    assertEquals("GET", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("value", response.getStringValue("string"));
    assertEquals(42, response.getLongValue("long").intValue());
    assertEquals(Math.PI, response.getDoubleValue("double").doubleValue());
    assertTrue(response.getBooleanValue("boolean"));
  }

  /** test a simple get without response object. */
  public void testGetNoResponse() throws Exception {
    final AnyMap response = _client.get(_mockPath + NORESPONSE_PARAM);
    assertNull(response);
  }

  /** test a simple get without parameters. */
  public void testPost() throws Exception {
    final AnyMap response = _client.post(_mockPath);
    assertNotNull(response);
    assertEquals(2, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
  }

  /** test a post without response object. */
  public void testPostNoResponse() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("noResponse", "x");
    final AnyMap response = _client.post(_mockPath, parameters);
    assertNull(response);
  }

  /** test a post with parameters. */
  public void testPostWithParameters() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    parameters.put("long", 42);
    parameters.put("double", Math.PI);
    parameters.put("boolean", true);
    final AnyMap response = _client.post(_mockPath, parameters);
    assertNotNull(response);
    assertEquals(6, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("value", response.getStringValue("string"));
    assertEquals(42, response.getLongValue("long").intValue());
    assertEquals(Math.PI, response.getDoubleValue("double").doubleValue());
    assertTrue(response.getBooleanValue("boolean"));
  }

  /** test a post with parameters. */
  public void testPostWithReponseHeaderResult() throws Exception {
    final Record record = DataFactory.DEFAULT.createRecord("testPostRecord");
    record.getMetadata().put("string", "value");
    final AnyMap responseHeaderFields = DataFactory.DEFAULT.createAnyMap();
    final AnyMap response = _client.postWithResponseHeader(_mockPath, record, responseHeaderFields);
    assertNotNull(response);
    assertNotSame(0, responseHeaderFields.size());
    assertEquals(4, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("testPostRecord", response.getStringValue(Record.RECORD_ID));
    assertEquals("value", response.getStringValue("string"));
  }

  /** test a post with parameters as string. */
  public void testPostWithStringParameters() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    parameters.put("long", 42);
    parameters.put("double", Math.PI);
    parameters.put("boolean", true);
    final IpcAnyWriter writer = new IpcAnyWriter();
    final AnyMap response =
      _client.post(_mockPath, writer.writeJsonObject(parameters), HttpRequestFactory.ENCODING,
        HttpRequestFactory.MIMETYPE_JSON);
    assertNotNull(response);
    assertEquals(6, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("value", response.getStringValue("string"));
    assertEquals(42, response.getLongValue("long").intValue());
    assertEquals(Math.PI, response.getDoubleValue("double").doubleValue());
    assertTrue(response.getBooleanValue("boolean"));
  }

  /** test a post with parameters. */
  public void testPutWithParameters() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    parameters.put("long", 42);
    parameters.put("double", Math.PI);
    parameters.put("boolean", true);
    final AnyMap response = _client.put(_mockPath, parameters);
    assertNotNull(response);
    assertEquals(6, response.size());
    assertEquals("PUT", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("value", response.getStringValue("string"));
    assertEquals(42, response.getLongValue("long").intValue());
    assertEquals(Math.PI, response.getDoubleValue("double").doubleValue());
    assertTrue(response.getBooleanValue("boolean"));
  }

  /** test a post without response object. */
  public void testPutNoResponse() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("noResponse", "x");
    final AnyMap response = _client.put(_mockPath, parameters);
    assertNull(response);
  }

  /** test a simple get without response object. */
  public void testDelete() throws Exception {
    final AnyMap response = _client.delete(_mockPath);
    assertNotNull(response);
    assertEquals(2, response.size());
    assertEquals("DELETE", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
  }

  /** test a delete with parameters. */
  public void testDeleteWithParameters() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    parameters.put("long", 42);
    parameters.put("double", Math.PI);
    parameters.put("boolean", true);
    final AnyMap response = _client.delete(_mockPath, parameters);
    assertNotNull(response);
    assertEquals(6, response.size());
    assertEquals("DELETE", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("value", response.getStringValue("string"));
    assertEquals(42, response.getLongValue("long").intValue());
    assertEquals(Math.PI, response.getDoubleValue("double").doubleValue());
    assertTrue(response.getBooleanValue("boolean"));
  }

  /** test a simple get without response object. */
  public void testDeleteNoResponse() throws Exception {
    final AnyMap response = _client.delete(_mockPath + NORESPONSE_PARAM);
    assertNull(response);
  }

  /** test request with one attachment created from a byte array. */
  public void testByteArrayAttachment() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    final AttachmentWrapper attachments =
      new AttachmentWrapper("attachmentName", "attachmentValue".getBytes("utf-8"));
    final AnyMap response = _client.post(_mockPath, parameters, attachments);
    assertNotNull(response);
    assertEquals(4, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("value", response.getStringValue("string"));
    assertEquals("attachmentValue", response.getStringValue("attachmentName"));
  }

  /** test request with one attachment created from a string. */
  public void testStringAttachment() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    final AttachmentWrapper attachments = new AttachmentWrapper("attachmentName", "attachmentValue");
    final AnyMap response = _client.post(_mockPath, parameters, attachments);
    assertNotNull(response);
    assertEquals(4, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("value", response.getStringValue("string"));
    assertEquals("attachmentValue", response.getStringValue("attachmentName"));
  }

  /** test request with one attachment created from an input stream. */
  public void testStreamAttachment() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    final AttachmentWrapper attachments =
      new AttachmentWrapper("attachmentName", new ByteArrayInputStream("attachmentValue".getBytes("utf-8")));
    final AnyMap response = _client.post(_mockPath, parameters, attachments);
    assertNotNull(response);
    assertEquals(4, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("value", response.getStringValue("string"));
    assertEquals("attachmentValue", response.getStringValue("attachmentName"));
  }

  /** test request with one attachment created from a file. */
  public void testFileAttachment() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    final File attachmentFile = File.createTempFile("testFileAttachment", ".txt");
    FileUtils.write(attachmentFile, "attachmentValue");
    attachmentFile.deleteOnExit();
    final AttachmentWrapper attachments = new AttachmentWrapper("attachmentName", attachmentFile);
    final AnyMap response = _client.post(_mockPath, parameters, attachments);
    assertNotNull(response);
    assertEquals(4, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("value", response.getStringValue("string"));
    assertEquals("attachmentValue", response.getStringValue("attachmentName"));
  }

  /** test request with multiple attachments created from different objects. */
  public void testMultipleAttachments() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    final File attachmentFile = File.createTempFile("testFileAttachment", ".txt");
    FileUtils.write(attachmentFile, "attachmentFileValue");
    attachmentFile.deleteOnExit();
    final AttachmentWrapper attachments = new AttachmentWrapper();
    attachments.add("attachmentBytes", "attachmentByteValue".getBytes("utf-8"));
    attachments.add("attachmentString", "attachmentStringValue");
    attachments.add("attachmentFile", attachmentFile);
    final InputStream attachmentStream = new FileInputStream(attachmentFile);
    try {
      attachments.add("attachmentStream", attachmentFile);
      final AnyMap response = _client.post(_mockPath, parameters, attachments);
      assertNotNull(response);
      assertEquals(7, response.size());
      assertEquals("POST", response.getStringValue("method"));
      assertEquals(_mockPath, response.getStringValue("uri"));
      assertEquals("value", response.getStringValue("string"));
      assertEquals("attachmentByteValue", response.getStringValue("attachmentBytes"));
      assertEquals("attachmentStringValue", response.getStringValue("attachmentString"));
      assertEquals("attachmentFileValue", response.getStringValue("attachmentStream"));
      assertEquals("attachmentFileValue", response.getStringValue("attachmentFile"));
    } finally {
      IOUtils.closeQuietly(attachmentStream);
    }
  }

  /** test request by posting a record without attachments. */
  public void testPostRecord() throws Exception {
    final Record record = DataFactory.DEFAULT.createRecord("testPostRecord");
    record.getMetadata().put("string", "value");
    final AnyMap response = _client.post(_mockPath, record);
    assertNotNull(response);
    assertEquals(4, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("testPostRecord", response.getStringValue(Record.RECORD_ID));
    assertEquals("value", response.getStringValue("string"));
  }

  /** test request by posting a record with attachments. */
  public void testPostRecordWithAttachments() throws Exception {
    final Record record = DataFactory.DEFAULT.createRecord("testPostRecordWithAttachments");
    record.getMetadata().put("string", "value");
    record.setAttachment("attachmentName", "attachmentValue".getBytes("utf-8"));
    final AnyMap response = _client.post(_mockPath, record);
    assertNotNull(response);
    assertEquals(5, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("testPostRecordWithAttachments", response.getStringValue(Record.RECORD_ID));
    assertEquals("value", response.getStringValue("string"));
    assertEquals("attachmentValue", response.getStringValue("attachmentName"));
  }

  /** test request by putting a record without attachments. */
  public void testPutRecord() throws Exception {
    final Record record = DataFactory.DEFAULT.createRecord("testPutRecord");
    record.getMetadata().put("string", "value");
    final AnyMap response = _client.put(_mockPath, record);
    assertNotNull(response);
    assertEquals(4, response.size());
    assertEquals("PUT", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("testPutRecord", response.getStringValue(Record.RECORD_ID));
    assertEquals("value", response.getStringValue("string"));
  }

  /** test request by putting a record with attachments (no attachments are transmitted in PUT requests). */
  public void testPutRecordWithAttachments() throws Exception {
    final Record record = DataFactory.DEFAULT.createRecord("testPutRecordWithAttachments");
    record.getMetadata().put("string", "value");
    record.setAttachment("attachmentName", "attachmentValue".getBytes("utf-8"));
    final AnyMap response = _client.put(_mockPath, record);
    assertNotNull(response);
    assertEquals(4, response.size());
    assertEquals("PUT", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("testPutRecordWithAttachments", response.getStringValue(Record.RECORD_ID));
    assertEquals("value", response.getStringValue("string"));
  }

  /** test creating a POST request from in JSON input stream. */
  public void testPostRequestFromStream() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("string", "value");
    parameters.put("long", 42);
    parameters.put("double", Math.PI);
    parameters.put("boolean", true);
    final byte[] bytes = new IpcAnyWriter(true).writeJsonObject(parameters).getBytes("utf-8");
    final AnyMap response =
      _client.invoke(HttpMethod.POST, _mockPath, new ByteArrayInputStream(bytes), null).asMap();
    assertNotNull(response);
    assertEquals(6, response.size());
    assertEquals("POST", response.getStringValue("method"));
    assertEquals(_mockPath, response.getStringValue("uri"));
    assertEquals("value", response.getStringValue("string"));
    assertEquals(42, response.getLongValue("long").intValue());
    assertEquals(Math.PI, response.getDoubleValue("double").doubleValue());
    assertTrue(response.getBooleanValue("boolean"));
  }

  /** test get method that responses with an error. */
  public void testGetError() throws Exception {
    try {
      _client.get(_mockPath + "?errorstatus=404");
      fail("exception expected.");
    } catch (final RestException ex) {
      assertEquals(404, ex.getResponseCode());
      assertNotNull(ex.getResponseObject());
    }
  }

  /** test delete method that responses with an error. */
  public void testDeleteError() throws Exception {
    try {
      _client.delete(_mockPath + "?errorstatus=503&errorreason=internal+error");
      fail("exception expected.");
    } catch (final RestException ex) {
      assertEquals(503, ex.getResponseCode());
      assertEquals("internal error", ex.getMessage());
      assertNotNull(ex.getResponseObject());
      assertTrue(ex.getResponseObject().containsKey("message"));
    }
  }

  /** test post method that responses with an error. */
  public void testPostError() throws Exception {
    try {
      final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
      parameters.put("errorstatus", 400);
      parameters.put("errorreason", "bad request");
      _client.post(_mockPath, parameters);
      fail("exception expected.");
    } catch (final RestException ex) {
      assertEquals(400, ex.getResponseCode());
      assertEquals("bad request", ex.getMessage());
      assertNotNull(ex.getResponseObject());
      assertTrue(ex.getResponseObject().containsKey("message"));
    }
  }

  /** test put method that responses with an error. */
  public void testPutError() throws Exception {
    try {
      final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
      parameters.put("errorstatus", 500);
      _client.post(_mockPath, parameters);
      fail("exception expected.");
    } catch (final RestException ex) {
      assertEquals(500, ex.getResponseCode());
      assertNotNull(ex.getResponseObject());
    }
  }

  /** test bulk response parsing. */
  public void testBulkResponse() throws Exception {
    final int expectedCount = 10;
    final BulkResponse bulk = _client.getBulk(_mockPath + "?bulksize=" + expectedCount, null);
    try {
      int count = 0;
      while (bulk.hasNext()) {
        final AnyMap bulkElement = bulk.next();
        assertEquals(count, bulkElement.getLongValue("bulkposition").intValue());
        count++;
      }
      assertEquals(expectedCount, count);
    } finally {
      bulk.close();
    }
  }

  /** test error in bulk requests. */
  public void testBulkError() throws Exception {
    try {
      _client.getBulk(_mockPath + "?errorstatus=404", null);
      fail("exception expected.");
    } catch (final RestException ex) {
      assertEquals(404, ex.getResponseCode());
      assertNotNull(ex.getResponseObject());
    }
  }

  /** test using an URI that is not correctly encoded. */
  public void testGetInvalidUri() throws Exception {
    try {
      _client.getBulk(_mockPath + "?invalid=param value", null);
      fail("exception expected.");
    } catch (final IllegalArgumentException ex) {
      ;
    }
  }

  public void testRequestWithSequence() throws Exception {
    final AnySeq seq = DataFactory.DEFAULT.createAnySeq();
    seq.add("one");
    seq.add(2);
    seq.add(3.4);
    final Any response = _client.invoke(HttpMethod.POST, _anyPath, seq, null, null);
    assertTrue(response.isMap());
    assertEquals("POST", response.asMap().getStringValue("method"));
    assertEquals(_anyPath, response.asMap().getStringValue("resource"));
    assertEquals(seq, response.asMap().get("data"));
  }

  /** test authentication via different port. */
  public void testAuthentication() throws Exception {
    final RestClient authClient = new DefaultRestClient("localhost:8081");
    try {
      authClient.get(_mockPath);
      fail();
    } catch (Exception e) {
      e.printStackTrace();
      assertTrue(e.getMessage().contains("401"));
    }
    authClient.setAuthParameters("admin", "padmin");
    final AnyMap response = authClient.get(_mockPath);
    assertNotNull(response);
  }

}
