/*********************************************************************************************************************
 * Copyright (c) 2008, 2014 Attensity Europe GmbH, Empolis Information Management GmbH and brox IT Solutions GmbH.
 * All rights reserved. This program and the accompanying materials are made available under the terms of the Eclipse
 * Public License v1.0 which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file.test;

import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;

/**
 * Helper class that creates a folder structure for testing.
 *
 * @author stuc07
 *
 */
public final class FilesystemHelper {

  public static Path initTestFilesystem() throws IOException {
    final Path rootFolder = Files.createTempDirectory("importing-crawler");

    final Path folderA = rootFolder.resolve("a");
    Files.createDirectories(folderA.resolve("aa"));
    Files.createDirectories(folderA.resolve("ab"));
    Files.createDirectories(rootFolder.resolve("b"));
    Files.write(rootFolder.resolve("c"), new byte[0]);
    Files.write(rootFolder.resolve("d.e"), new byte[0]);

    return rootFolder;
  }

  public static Path initComplexTestFilesystem() throws IOException {
    final Path rootFolder = Files.createTempDirectory("importing-crawler");

    createFiles(rootFolder, 10);
    createFolders(rootFolder, 5, 10, 2);

    return rootFolder;
  }

  public static Path initComplexTestFilesystemDeepFolderStructure() throws IOException {
    final Path rootFolder = Files.createTempDirectory("importing-crawler-deep");

    createFiles(rootFolder, 1);
    createFolders(rootFolder, 2, 1, 5); // -> 63 files

    return rootFolder;
  }

  private static void createFiles(final Path rootFolder, final int maxFileCount) throws IOException {
    Files.createDirectories(rootFolder);
    for (int i = 0; i < maxFileCount; i++) {
      Files.write(rootFolder.resolve("file" + i), new byte[0]);
    }
  }

  private static void createFolders(final Path rootFolder, final int maxFolderCount, final int maxFileCount,
    final int levels) throws IOException {
    for (int i = 0; i < maxFolderCount; i++) {
      final Path folder = rootFolder.resolve("folder" + i);
      createFiles(folder, maxFileCount);
      if (levels > 1) {
        final int subLevels = levels - 1;
        createFolders(folder, maxFolderCount, maxFileCount, subLevels);
      }
    }
  }

  public static void deleteDirectory(final Path dir) throws IOException {
    if (dir != null && Files.exists(dir)) {
      Files.walkFileTree(dir, new DeleteFileVisitor());
    }
  }

  public static void copyDirectory(final Path configDir, final Path workspaceDir, final String ignoreGlob)
    throws IOException {
    final CopyFileVisitor cfv = new CopyFileVisitor(configDir, workspaceDir, ignoreGlob);
    Files.walkFileTree(configDir, cfv);
  }

  private static final class DeleteFileVisitor extends SimpleFileVisitor<Path> {
    @Override
    public FileVisitResult postVisitDirectory(final Path dir, final IOException exc) throws IOException {
      Files.deleteIfExists(dir);
      return super.postVisitDirectory(dir, exc);
    }

    @Override
    public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) throws IOException {
      Files.deleteIfExists(file);
      return super.visitFile(file, attrs);
    }

  }

  private static final class CopyFileVisitor extends SimpleFileVisitor<Path> {
    private final Path _src;

    private final Path _dst;

    private final PathMatcher _ignoreMatcher;

    public CopyFileVisitor(final Path src, final Path dst, final String ignoreGlob) {
      _src = src;
      _dst = dst;
      _ignoreMatcher = FileSystems.getDefault().getPathMatcher("glob:" + ignoreGlob);
    }

    @Override
    public FileVisitResult preVisitDirectory(final Path dir, final BasicFileAttributes attrs) throws IOException {
      if (!_ignoreMatcher.matches(dir)) {
        Files.copy(dir, _dst.resolve(_src.relativize(dir)), StandardCopyOption.COPY_ATTRIBUTES,
          StandardCopyOption.REPLACE_EXISTING);
        return super.preVisitDirectory(dir, attrs);
      } else {
        return FileVisitResult.SKIP_SUBTREE;
      }
    }

    @Override
    public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) throws IOException {
      if (!_ignoreMatcher.matches(file)) {
        Files.copy(file, _dst.resolve(_src.relativize(file)), StandardCopyOption.COPY_ATTRIBUTES,
          StandardCopyOption.REPLACE_EXISTING);
      }
      return super.visitFile(file, attrs);
    }

  }

}
