/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file.test;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collection;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.file.FileCrawlerService;
import org.eclipse.smila.importing.util.FilePathNormalizer;
import org.eclipse.smila.importing.util.PropertyNameMapper;
import org.eclipse.smila.jobmanager.JobRunDataProvider;
import org.eclipse.smila.jobmanager.JobRunEngine;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Test for FileFetcherWorker.
 */
public class TestFileFetcherWorker extends DeclarativeServiceTestCase {

  private static final int DEFAULT_WAIT_TIME = 20000;

  private static final String DATA_SOURCE = "fetchDocument";

  private static final String STORE = "records";

  private static final String BUCKET = "fetchedFiles";

  private static final String WORKFLOWNAME = "fetchDocument";

  private JobRunEngine _jobRunEngine;

  private JobRunDataProvider _jobRunDataProvider;

  private BulkbuilderService _bulkBuilder;

  private ObjectStoreService _objectStore;

  private DefinitionPersistence _definitionPersistence;

  private final PropertyNameMapper _mapper = new PropertyNameMapper(initMapping());

  @Override
  public void setUp() throws Exception {
    _jobRunEngine = getService(JobRunEngine.class);
    _jobRunDataProvider = getService(JobRunDataProvider.class);
    _bulkBuilder = getService(BulkbuilderService.class);
    _objectStore = getService(ObjectStoreService.class);
    _objectStore.ensureStore(STORE);
    _objectStore.clearStore(STORE);
    _definitionPersistence = getService(DefinitionPersistence.class);
  }

  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
  }

  /** test crawling with a simple configuration. */
  public void testSimpleCrawling() throws Exception {
    final Path file = Paths.get("./configuration/filter/maxSize/10.txt");
    final JobDefinition jobDefinition = createJob();
    _definitionPersistence.addJob(jobDefinition);
    final String jobRunId = _jobRunEngine.startJob(getName());
    pushRecord(getName(), file);
    _jobRunEngine.finishJob(getName(), jobRunId);
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);
    checkBulks(1);
    checkFetchedRecord(file);
  }

  /** check number of bulks. */
  private void checkBulks(final int expectedNumberOfBulks) throws Exception {
    final Collection<StoreObject> objects = _objectStore.getStoreObjectInfos(STORE, BUCKET);
    assertEquals(expectedNumberOfBulks, objects.size());
  }

  /** checks the fetched record. */
  private void checkFetchedRecord(final Path file) throws ObjectStoreException, IOException {
    final Collection<StoreObject> objects = _objectStore.getStoreObjectInfos(STORE, BUCKET);
    assertFalse(objects.isEmpty());
    for (final StoreObject objectInfo : objects) {
      final InputStream bulkStream = _objectStore.readObject(STORE, objectInfo.getId());
      try {
        final BinaryObjectStreamIterator records = new BinaryObjectStreamIterator(bulkStream);
        final Record record = records.next();
        assertNotNull(record);
        assertNotNull(record.getId());
        assertEquals(DATA_SOURCE, record.getSource());
        final AnyMap metadata = record.getMetadata();

        // assert that no compound properties are set
        assertFalse(metadata.containsKey(ImportingConstants.ATTRIBUTE_COMPOUNDPATH));
        assertFalse(metadata.containsKey(ImportingConstants.ATTRIBUTE_COMPOUNDFLAG));

        // check some values (not last modified, may change on checkout)
        assertTrue(metadata.get(_mapper.get(FileCrawlerService.PROPERTY_FILE_LAST_MODIFIED).get(0)).isDateTime());
        assertEquals(file.toRealPath().toString(),
          metadata.getStringValue(_mapper.get(FileCrawlerService.PROPERTY_FILE_PATH).get(0)));
        assertEquals(FilePathNormalizer.getNormalizedPath(file.getParent().toRealPath().toString()),
          metadata.getStringValue(_mapper.get(FileCrawlerService.PROPERTY_FILE_FOLDER).get(0)));
        assertEquals(file.getFileName().toString(),
          metadata.getStringValue(_mapper.get(FileCrawlerService.PROPERTY_FILE_NAME).get(0)));
        assertEquals("txt", metadata.getStringValue(_mapper.get(FileCrawlerService.PROPERTY_FILE_EXTENSION).get(0)));
        assertTrue(metadata.get(_mapper.get(FileCrawlerService.PROPERTY_FILE_SIZE).get(0)).isLong());
        assertEquals(Files.size(file),
          metadata.getLongValue(_mapper.get(FileCrawlerService.PROPERTY_FILE_SIZE).get(0)).intValue());
        assertTrue(record.hasAttachments());
        assertEquals("aaaaaaaaaa",
          new String(record.getAttachmentAsBytes(_mapper.get(FileCrawlerService.ATTACHMENT_FILE_CONTENT).get(0))));
      } finally {
        IOUtils.closeQuietly(bulkStream);
      }
    }
  }

  private JobDefinition createJob() throws Exception {
    final AnyMap job = DataFactory.DEFAULT.createAnyMap();
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("tempStore", STORE);
    parameters.put("store", STORE);
    parameters.put(ImportingConstants.TASK_PARAM_DATA_SOURCE, DATA_SOURCE);
    parameters.put(ImportingConstants.TASK_PARAM_MAPPING, initMapping());
    job.put(JobDefinition.KEY_PARAMETERS, parameters);
    job.put(JobDefinition.KEY_NAME, getName());
    job.put(JobDefinition.KEY_WORKFLOW, WORKFLOWNAME);
    return new JobDefinition(job);
  }

  protected void waitForJobRunCompleted(final String jobName, final String jobId, final long maxWaitTime)
    throws Exception {
    waitForJobRun(jobName, jobId, maxWaitTime, JobState.SUCCEEDED);
  }

  /** Waits for a job to be completed. */
  protected void waitForJobRun(final String jobName, final String jobId, final long maxWaitTime,
    final JobState expectedJobState) throws Exception {
    final long sleepTime = 500L;
    final long millisStarted = System.currentTimeMillis();
    while (true) {
      final AnyMap runData = _jobRunDataProvider.getJobRunData(jobName, jobId);
      final String jobRunState = runData.getStringValue(JobManagerConstants.DATA_JOB_STATE);
      if (jobRunState != null) {
        final JobState state = JobState.valueOf(jobRunState);
        if (state == expectedJobState) {
          return; // finally found what we're waiting for.
        }
        if (expectedJobState == JobState.SUCCEEDED) {
          assertNotSame(JobState.FAILED, state);
        }
      }
      assertTrue("Waited too long for job to complete", System.currentTimeMillis() - millisStarted <= maxWaitTime);
      Thread.sleep(sleepTime);
    }
  }

  /** initializes mapping. */
  private AnyMap initMapping() {
    final AnyMap map = DataFactory.DEFAULT.createAnyMap();
    map.put(FileCrawlerService.PROPERTY_FILE_EXTENSION, "my-file-extension");
    map.put(FileCrawlerService.PROPERTY_FILE_FOLDER, "my-file-folder");
    map.put(FileCrawlerService.PROPERTY_FILE_LAST_MODIFIED, "my-file-last-modified");
    map.put(FileCrawlerService.PROPERTY_FILE_NAME, "my-file-name");
    map.put(FileCrawlerService.PROPERTY_FILE_PATH, "my-file-path");
    map.put(FileCrawlerService.PROPERTY_FILE_SIZE, "my-file-size");
    map.put(FileCrawlerService.ATTACHMENT_FILE_CONTENT, "my-file-content");
    return map;
  }

  private void pushRecord(final String jobName, final Path file) throws Exception {
    final Record record = DataFactory.DEFAULT.createRecord(file.toRealPath().toString(), DATA_SOURCE);
    record.getMetadata().put(_mapper.get(FileCrawlerService.PROPERTY_FILE_PATH).get(0),
      file.toRealPath().toString());
    _bulkBuilder.addRecord(jobName, record);
    _bulkBuilder.commitJob(jobName);
  }

}
