/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.jdbc.test;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collection;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.eclipse.smila.importing.DeltaImportStrategy;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.jdbc.JdbcCrawlerWorker;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.StoreObject;

public class TestJdbcCrawlerWorker extends JdbcCrawlerJobTestBase {

  private static final String BUCKET = "crawledRows";

  private static final String WORKFLOWNAME = "crawlDatabase";

  /** test crawling with a simple configuration. */
  public void testSimpleCrawling() throws Exception {
    final int maxRecordsPerBulk = 1000;
    final AnyMap jobDefinition =
      createJob(getConnectUrl(), "SELECT * FROM " + DB_TABLE_NAME, null, maxRecordsPerBulk);
    defineJob(jobDefinition);
    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);

    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), maxRecordsPerBulk), BUCKET);
    final Collection<String> expectedAttributes =
      Arrays.asList(ImportingConstants.ATTRIBUTE_DELTA_HASH, "my-int-value", "my-big-int-value", "my-double-value",
        "my-varchar-value");
    final Collection<String> unexpectedAttributes = Arrays.asList("float_val");
    final Collection<String> expectedAttachments = Arrays.asList("my-binary-attachment");
    checkCrawledRecords(getNumberOfRowsToCreate(), BUCKET, expectedAttributes, unexpectedAttributes,
      expectedAttachments);
  }

  /** test crawling where 'deltaColumns' parameter includes a non-existing column. */
  public void testCrawlingWithDeltaColumnNullValues() throws Exception {
    final int maxRecordsPerBulk = 1000;
    final AnyMap jobDefinition =
      createJob(getConnectUrl(), "SELECT * FROM " + DB_TABLE_NAME, null, maxRecordsPerBulk);
    jobDefinition.getMap("parameters").getSeq(JdbcCrawlerWorker.TASK_PARAM_DELTA_COLUMNS, true).add("null-column");
    defineJob(jobDefinition);
    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);

    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), maxRecordsPerBulk), BUCKET);
    final Collection<String> expectedAttributes =
      Arrays.asList(ImportingConstants.ATTRIBUTE_DELTA_HASH, "my-int-value", "my-big-int-value", "my-double-value",
        "my-varchar-value");
    final Collection<String> unexpectedAttributes = Arrays.asList("float_val");
    final Collection<String> expectedAttachments = Arrays.asList("my-binary-attachment");
    checkCrawledRecordsDeltaNull(getNumberOfRowsToCreate(), BUCKET, expectedAttributes, unexpectedAttributes,
      expectedAttachments);
  }

  /** test delta hash creation depending on 'deltaColumns' and 'deltaImportStrategy' parameter. */
  public void testDeltaHash() throws Exception {
    final AnyMap jobDefinition = createJob(getConnectUrl(), "SELECT * FROM " + DB_TABLE_NAME, null, 1000);
    final int expectedNumberOfRecords = getNumberOfRowsToCreate();

    // test with deltaColumns set and deltaImportStrategy != DISABLED -> delta hash is created from delta columns
    String delta1;
    {
      super.setUp(); // clear store
      final AnySeq deltaColumns = DataFactory.DEFAULT.createAnySeq();
      deltaColumns.add("bigint_val");
      jobDefinition.getMap("parameters").put(JdbcCrawlerWorker.TASK_PARAM_DELTA_COLUMNS, deltaColumns);
      jobDefinition.getMap("parameters").put("deltaImportStrategy", DeltaImportStrategy.FULL.name());
      defineJob(jobDefinition);
      final String jobRunId = _jobRunEngine.startJob(getName());
      waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);
      delta1 = checkCrawledRecordsDeltaHash(expectedNumberOfRecords, BUCKET, true);
      assertNotNull(delta1);
      assertFalse(delta1.isEmpty());
    }

    // test with deltaColumns not set and deltaImportStrategy != DISABLED -> delta hash is created from mapping columns
    String delta2;
    {
      super.setUp(); // clear store
      jobDefinition.getMap("parameters").remove(JdbcCrawlerWorker.TASK_PARAM_DELTA_COLUMNS);
      jobDefinition.getMap("parameters").put("deltaImportStrategy", DeltaImportStrategy.FULL.name());
      defineJob(jobDefinition);
      final String jobRunId = _jobRunEngine.startJob(getName());
      waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);
      delta2 = checkCrawledRecordsDeltaHash(expectedNumberOfRecords, BUCKET, true);
      assertNotNull(delta2);
      assertFalse(delta2.isEmpty());
    }
    assertFalse(delta1.equals(delta2));

    // test with deltaImportStrategy DISABLED -> no delta hash is created
    {
      super.setUp(); // clear store
      final AnySeq deltaColumns = DataFactory.DEFAULT.createAnySeq();
      deltaColumns.add("bigint_val");
      jobDefinition.getMap("parameters").put(JdbcCrawlerWorker.TASK_PARAM_DELTA_COLUMNS, deltaColumns);
      jobDefinition.getMap("parameters").put("deltaImportStrategy", DeltaImportStrategy.DISABLED.name());
      defineJob(jobDefinition);
      final String jobRunId = _jobRunEngine.startJob(getName());
      waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);
      checkCrawledRecordsDeltaHash(expectedNumberOfRecords, BUCKET, false);
    }

    // test with 'deltaColumns' set to empty list -> no delta hash is created
    {
      super.setUp(); // clear store
      final AnySeq deltaColumns = DataFactory.DEFAULT.createAnySeq();
      jobDefinition.getMap("parameters").put(JdbcCrawlerWorker.TASK_PARAM_DELTA_COLUMNS, deltaColumns);
      jobDefinition.getMap("parameters").put("deltaImportStrategy", DeltaImportStrategy.FULL.name());
      defineJob(jobDefinition);
      final String jobRunId = _jobRunEngine.startJob(getName());
      waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);
      checkCrawledRecordsDeltaHash(expectedNumberOfRecords, BUCKET, false);
    }
  }

  /** test 'maxAttachmentSize' parameter. */
  public void testMaxAttachmentSize() throws Exception {
    final int maxRecordsPerBulk = 1000;
    final AnyMap jobDefinition =
      createJob(getConnectUrl(), "SELECT * FROM " + DB_TABLE_NAME, null, maxRecordsPerBulk);
    // set max. attachment size to 1 byte -> attachment of the test will not be accepted
    jobDefinition.getMap("parameters").put(JdbcCrawlerWorker.TASK_PARAM_MAX_ATTACHMENT_SIZE, 1);
    defineJob(jobDefinition);

    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);

    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), maxRecordsPerBulk), BUCKET);
    final Collection<String> expectedAttributes =
      Arrays.asList(ImportingConstants.ATTRIBUTE_DELTA_HASH, "my-int-value", "my-big-int-value", "my-double-value",
        "my-varchar-value");
    final Collection<String> unexpectedAttributes = Arrays.asList("float_val");
    final Collection<String> expectedAttachments = null; // -> test fails if records have attachments
    checkCrawledRecords(getNumberOfRowsToCreate(), BUCKET, expectedAttributes, unexpectedAttributes,
      expectedAttachments);
  }

  protected String checkCrawledRecordsDeltaHash(final int expectedNumberOfRecords, final String bucket,
    final boolean isDeltaHashExpected) throws ObjectStoreException, IOException {
    String deltaHash = null;
    final Collection<StoreObject> objects = _objectStore.getStoreObjectInfos(STORE, bucket);
    assertFalse(objects.isEmpty());
    int recordCount = 0;
    for (final StoreObject objectInfo : objects) {
      final InputStream bulkStream = _objectStore.readObject(STORE, objectInfo.getId());
      final BinaryObjectStreamIterator records = new BinaryObjectStreamIterator(bulkStream);
      try {
        while (records.hasNext()) {
          final Record record = records.next();
          recordCount++;
          deltaHash = record.getMetadata().getStringValue(ImportingConstants.ATTRIBUTE_DELTA_HASH);
          if (isDeltaHashExpected) {
            assertNotNull(deltaHash);
          } else {
            assertNull(deltaHash);
          }
          return deltaHash;
        }
      } finally {
        records.close();
      }
    }
    assertEquals(expectedNumberOfRecords, recordCount);
    return deltaHash;
  }

  protected void checkCrawledRecordsDeltaNull(final int expectedNumberOfRecords, final String bucket,
    final Collection<String> expectedAttributes, final Collection<String> unexpectedAttributes,
    final Collection<String> expectedAttachments) throws ObjectStoreException, IOException {
    final Collection<StoreObject> objects = _objectStore.getStoreObjectInfos(STORE, bucket);
    int recordCount = 0;
    assertFalse(objects.isEmpty());
    for (final StoreObject objectInfo : objects) {
      final InputStream bulkStream = _objectStore.readObject(STORE, objectInfo.getId());
      final BinaryObjectStreamIterator records = new BinaryObjectStreamIterator(bulkStream);
      try {
        while (records.hasNext()) {
          final Record record = records.next();
          recordCount++;
          checkCrawledRecord(record, expectedAttributes, unexpectedAttributes, expectedAttachments);
          assertTrue(record.getMetadata().getStringValue(ImportingConstants.ATTRIBUTE_DELTA_HASH).endsWith("NULL"));
        }
      } finally {
        records.close();
      }
    }
    assertEquals(expectedNumberOfRecords, recordCount);
  }

  @Override
  protected AnyMap initMapping() {
    final AnyMap map = DataFactory.DEFAULT.createAnyMap();
    map.put("int_val", "my-int-value");
    map.put("bigint_val", "my-big-int-value");
    map.put("double_val", "my-double-value");
    // map.put("float_val", "my-file-extension"); // don't map to check if the value is excluded in the result record
    map.put("varchar_val", "my-varchar-value");
    map.put("varbinary_val", "my-binary-attachment");
    return map;
  }

  @Override
  protected String getWorkflow() {
    return WORKFLOWNAME;
  }

  /** @return the number of rows to create. */
  @Override
  protected int getNumberOfRowsToCreate() {
    return 10000;
  }

}
