/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.jdbc.test;

import java.util.Arrays;
import java.util.Collection;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.jdbc.JdbcCrawlerWorker;
import org.eclipse.smila.importing.crawler.jdbc.JdbcFetcherWorker;
import org.eclipse.smila.taskworker.Worker;
import org.eclipse.smila.utils.service.ServiceUtils;
import org.osgi.framework.ServiceReference;

public class TestJdbcFetcherWorker extends JdbcCrawlerJobTestBase {

  private static final int NO_OF_ROWS = 1000;

  private static final int MAX_RECORDS_PER_BULK = 100;

  private static final String OUT_BUCKET = "fetchedRows";

  private static final String WORKFLOWNAME = "crawlAndFetch";

  /** tests OSGI service. */
  @SuppressWarnings("rawtypes")
  public void testService() throws Exception {
    final ServiceReference[] services = ServiceUtils.getServiceReferences(Worker.class);
    assertTrue("no worker services started.", services.length > 0);
    for (final ServiceReference service : services) {
      final Worker worker = ServiceUtils.getService(service, Worker.class);
      if (worker instanceof JdbcFetcherWorker) {
        assertEquals(JdbcFetcherWorker.NAME, worker.getName());
        return; // worker found, test ok.
      }
    }
    fail("JdbcFetcherWorker not found");
  }

  /** tests workflow with fetcher, but fetching is not configured (no 'fetchSQL' parameter) -> Fetcher output = input */
  public void testWithoutFetching() throws Exception {
    final String crawlSql = "SELECT * FROM " + DB_TABLE_NAME;
    final AnyMap jobDefinition = createJob(getConnectUrl(), crawlSql, null, MAX_RECORDS_PER_BULK);
    defineJob(jobDefinition);
    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);
    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), MAX_RECORDS_PER_BULK), OUT_BUCKET);
  }

  /**
   * tests workflow with fetcher and real fetching ('fetchSQL' parameter set) by using fetch parameter attribute with
   * one value.
   */
  public void testWithFetchingAndFetchParamAttribute_singleValue() throws Exception {
    final String crawlSql = "SELECT int_val, bigint_val FROM " + DB_TABLE_NAME;
    final String fetchSql = "SELECT varchar_val, varbinary_val FROM " + DB_TABLE_NAME + " WHERE int_val = ?";

    final AnyMap jobDefinition = createJob(getConnectUrl(), crawlSql, fetchSql, MAX_RECORDS_PER_BULK);
    final Value fetchParameterAttributes = DataFactory.DEFAULT.createStringValue("my-int-value");
    jobDefinition.getMap("parameters").put(JdbcFetcherWorker.TASK_PARAM_FETCH_PARAM_ATTRIBUTES,
      fetchParameterAttributes);
    defineJob(jobDefinition);
    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);

    final Collection<String> expectedAttributes =
      Arrays
        .asList(ImportingConstants.ATTRIBUTE_DELTA_HASH, "my-int-value", "my-big-int-value", "my-varchar-value");
    final Collection<String> unexpectedAttributes = Arrays.asList("my-float-value", "my-double-value");
    final Collection<String> expectedAttachments = Arrays.asList("my-binary-attachment");
    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), MAX_RECORDS_PER_BULK), OUT_BUCKET);
    checkCrawledRecords(getNumberOfRowsToCreate(), OUT_BUCKET, expectedAttributes, unexpectedAttributes,
      expectedAttachments);
  }

  /**
   * tests workflow with fetcher and real fetching ('fetchSQL' parameter set) by using fetch parameter attribute with
   * list of values.
   */
  public void testWithFetchingAndFetchParamAttribute_listValue() throws Exception {
    final String crawlSql = "SELECT int_val, bigint_val FROM " + DB_TABLE_NAME;
    final String fetchSql =
      "SELECT varchar_val, varbinary_val FROM " + DB_TABLE_NAME + " WHERE int_val = ? and bigint_val = ?";

    final AnyMap jobDefinition = createJob(getConnectUrl(), crawlSql, fetchSql, MAX_RECORDS_PER_BULK);
    final AnySeq fetchParameterAttributes = DataFactory.DEFAULT.createAnySeq();
    fetchParameterAttributes.add("my-int-value");
    fetchParameterAttributes.add("my-big-int-value");
    jobDefinition.getMap("parameters").put(JdbcFetcherWorker.TASK_PARAM_FETCH_PARAM_ATTRIBUTES,
      fetchParameterAttributes);
    defineJob(jobDefinition);
    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);

    final Collection<String> expectedAttributes =
      Arrays
        .asList(ImportingConstants.ATTRIBUTE_DELTA_HASH, "my-int-value", "my-big-int-value", "my-varchar-value");
    final Collection<String> unexpectedAttributes = Arrays.asList("my-float-value", "my-double-value");
    final Collection<String> expectedAttachments = Arrays.asList("my-binary-attachment");
    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), MAX_RECORDS_PER_BULK), OUT_BUCKET);
    checkCrawledRecords(getNumberOfRowsToCreate(), OUT_BUCKET, expectedAttributes, unexpectedAttributes,
      expectedAttachments);
  }

  /**
   * tests workflow with fetcher and real fetching ('fetchSQL' parameter set) by using fetch parameter attribute with
   * one value.
   */
  public void testWithFetchingWithNullAttributeValues() throws Exception {
    final String crawlSql = "SELECT int_val, bigint_val FROM " + DB_TABLE_NAME;
    final String fetchSql =
      "SELECT bigint_val, varchar_val, varbinary_val FROM " + DB_TABLE_NAME + " WHERE int_val = ?";

    final AnyMap jobDefinition = createJob(getConnectUrl(), crawlSql, fetchSql, MAX_RECORDS_PER_BULK);
    final Value fetchParameterAttributes = DataFactory.DEFAULT.createStringValue("null-value");
    jobDefinition.getMap("parameters").put(JdbcFetcherWorker.TASK_PARAM_FETCH_PARAM_ATTRIBUTES,
      fetchParameterAttributes);
    defineJob(jobDefinition);
    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);

    final Collection<String> expectedAttributes =
      Arrays.asList(ImportingConstants.ATTRIBUTE_DELTA_HASH, "my-int-value", "my-big-int-value");
    final Collection<String> unexpectedAttributes =
      Arrays.asList("my-float-value", "my-double-value", "my-varchar-value");
    final Collection<String> expectedAttachments = null;
    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), MAX_RECORDS_PER_BULK), OUT_BUCKET);
    checkCrawledRecords(getNumberOfRowsToCreate(), OUT_BUCKET, expectedAttributes, unexpectedAttributes,
      expectedAttachments);
  }

  /** test 'maxAttachmentSize' parameter. */
  public void testMaxAttachmentSize() throws Exception {
    final String crawlSql = "SELECT int_val, bigint_val FROM " + DB_TABLE_NAME;
    final String fetchSql = "SELECT varchar_val, varbinary_val FROM " + DB_TABLE_NAME + " WHERE int_val = ?";

    final AnyMap jobDefinition = createJob(getConnectUrl(), crawlSql, fetchSql, MAX_RECORDS_PER_BULK);
    // set max. attachment size to 1 byte -> attachment of the test will not be accepted
    jobDefinition.getMap("parameters").put(JdbcCrawlerWorker.TASK_PARAM_MAX_ATTACHMENT_SIZE, 1);

    final Value fetchParameterAttributes = DataFactory.DEFAULT.createStringValue("my-int-value");
    jobDefinition.getMap("parameters").put(JdbcFetcherWorker.TASK_PARAM_FETCH_PARAM_ATTRIBUTES,
      fetchParameterAttributes);
    defineJob(jobDefinition);
    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);

    final Collection<String> expectedAttributes =
      Arrays
        .asList(ImportingConstants.ATTRIBUTE_DELTA_HASH, "my-int-value", "my-big-int-value", "my-varchar-value");
    final Collection<String> unexpectedAttributes = Arrays.asList("my-float-value", "my-double-value");
    final Collection<String> expectedAttachments = null;
    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), MAX_RECORDS_PER_BULK), OUT_BUCKET);
    checkCrawledRecords(getNumberOfRowsToCreate(), OUT_BUCKET, expectedAttributes, unexpectedAttributes,
      expectedAttachments);
  }

  @Override
  protected int getNumberOfRowsToCreate() {
    return NO_OF_ROWS;
  }

  @Override
  protected String getWorkflow() {
    return WORKFLOWNAME;
  }

  @Override
  protected AnyMap initMapping() {
    final AnyMap map = DataFactory.DEFAULT.createAnyMap();
    map.put("int_val", "my-int-value");
    map.put("bigint_val", "my-big-int-value");
    map.put("varchar_val", "my-varchar-value");
    map.put("varbinary_val", "my-binary-attachment");
    return map;
  }

  @Override
  protected Any getIdColumns() {
    final Value idColumn = DataFactory.DEFAULT.createStringValue("int_val");
    return idColumn;
  }

}
