/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.web.test.util;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.http.client.HttpMethod;
import org.eclipse.smila.http.client.RestClient;
import org.eclipse.smila.http.client.RestException;
import org.eclipse.smila.http.client.impl.DefaultRestClient;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Base test class for tests concerning robots.txt.
 */
public class RobotsTxtUtil extends DeclarativeServiceTestCase {

  public static final String BASE_URL = "http://localhost:8765";

  public static final String ROBOTS_PATH = "/robots.txt";

  private static final RestClient REST_CLIENT = new DefaultRestClient(BASE_URL);

  /**
   * @param userAgentDisallows
   *          key: User-Agent, value: List of Disallows.
   */
  public static void putRobotsTxt(final Map<String, List<String>> userAgentDisallows) throws Exception {
    final StringBuilder robotsTxt = new StringBuilder();
    for (final String userAgent : userAgentDisallows.keySet()) {
      robotsTxt.append("# -------------").append("\n\r"); // comment
      robotsTxt.append("User-agent: " + userAgent).append("\n");
      final List<String> disallows = userAgentDisallows.get(userAgent);
      for (final String disallow : disallows) {
        robotsTxt.append("Disallow: " + disallow).append("\r");
      }
    }
    final InputStream in = new ByteArrayInputStream(robotsTxt.toString().getBytes("UTF-8"));
    REST_CLIENT.invoke(HttpMethod.PUT, ROBOTS_PATH + '/', in, null);
  }

  /** delete current robots.txt. */
  public static void deleteRobotsTxt() throws Exception {
    REST_CLIENT.invoke(HttpMethod.DELETE, ROBOTS_PATH + '/', null, null);
  }

  /** use POST to provoke exceptions. */
  public static void postRobotsTxtException(final int httpCode) throws Exception {
    postRobotsTxtException(Integer.toString(httpCode));
  }

  /** use POST to provoke exceptions. */
  public static void postRobotsTxtException(final String httpCode) throws Exception {
    final InputStream in = new ByteArrayInputStream(httpCode.getBytes("UTF-8"));
    REST_CLIENT.invoke(HttpMethod.POST, ROBOTS_PATH + '/', in, null);
  }

  /** simple test run for the helper methods. */
  public void test() throws Exception {
    final Map<String, List<String>> m = new HashMap<>();
    m.put("SMILA-Agent", Arrays.asList("/path1", "path2"));
    m.put("IAS-Agent", Arrays.asList("/path3"));
    putRobotsTxt(m);
    REST_CLIENT.invoke(HttpMethod.GET, ROBOTS_PATH, null, null);
    deleteRobotsTxt();
    try {
      REST_CLIENT.invoke(HttpMethod.GET, ROBOTS_PATH, null, null);
    } catch (final RestException e) {
      assertEquals(404, e.getResponseCode());
      ; // expected
    }

    putRobotsTxt(m);
    postRobotsTxtException("401");
    try {
      REST_CLIENT.invoke(HttpMethod.GET, ROBOTS_PATH, null, null);
    } catch (final RestException e) {
      assertEquals(401, e.getResponseCode());
      ; // expected
    }
    postRobotsTxtException("403");
    try {
      REST_CLIENT.invoke(HttpMethod.GET, ROBOTS_PATH, null, null);
    } catch (final RestException e) {
      assertEquals(403, e.getResponseCode());
      ; // expected
    }
    postRobotsTxtException(""); // reset exception in handler
    REST_CLIENT.invoke(HttpMethod.GET, ROBOTS_PATH, null, null);
  }

}
