/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.state.objectstore.test;

import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.smila.importing.state.objectstore.StateStoreConfiguration;

/** Tests for the StateStoreConfiguration. */
public class TestStateStoreConfiguration extends TestCase {

  /** some digest string to test key generation with. */
  private static final String TEST_DIGEST = "0123456789abcdef";

  /** test default settings for key generation. */
  public void testDefaultConfiguration() {
    final Properties p = new Properties();
    final StateStoreConfiguration config = new StateStoreConfiguration(p);
    assertEquals("01/23/456789abcdef", config.getEntryKey(TEST_DIGEST));
  }

  /** test default settings for compound object key generation. */
  public void testCompoundIds() {
    final Properties p = new Properties();
    final StateStoreConfiguration config = new StateStoreConfiguration(p);
    assertEquals("AB/01/23456789abcdef", config.getEntryKey(TEST_DIGEST, "ABCDEF"));
  }

  /** test different shard.length setting. */
  public void testConfigureShardLength() {
    final Properties p = new Properties();
    p.put("shard.length", "3");
    final StateStoreConfiguration config = new StateStoreConfiguration(p);
    assertEquals("012/34/56789abcdef", config.getEntryKey(TEST_DIGEST));
  }

  /** test different key.pattern setting. */
  public void testConfigureKeyPattern() {
    final Properties p = new Properties();
    p.put("key.pattern", "%s.db/%s");
    final StateStoreConfiguration config = new StateStoreConfiguration(p);
    assertEquals("01.db/23/456789abcdef", config.getEntryKey(TEST_DIGEST));
  }

  /** test different segment.count setting. */
  public void testConfigureSegmentCount() {
    final Properties p = new Properties();
    p.put("segment.count", "2");
    final StateStoreConfiguration config = new StateStoreConfiguration(p);
    assertEquals("01/23/45/6789abcdef", config.getEntryKey(TEST_DIGEST));
  }

  /** test different segment.length setting. */
  public void testConfigureSegmentLength() {
    final Properties p = new Properties();
    p.put("segment.length", "3");
    final StateStoreConfiguration config = new StateStoreConfiguration(p);
    assertEquals("01/234/56789abcdef", config.getEntryKey(TEST_DIGEST));
  }

  /** test disabling segmentation via length=0 setting. */
  public void testDisableSegmentsViaLength() {
    final Properties p = new Properties();
    p.put("segment.length", "0");
    final StateStoreConfiguration config = new StateStoreConfiguration(p);
    assertEquals("01/23456789abcdef", config.getEntryKey(TEST_DIGEST));
  }

  /** test disabling segmentation via count=0 setting. */
  public void testDisableSegmentsViaCount() {
    final Properties p = new Properties();
    p.put("segment.count", "0");
    final StateStoreConfiguration config = new StateStoreConfiguration(p);
    assertEquals("01/23456789abcdef", config.getEntryKey(TEST_DIGEST));
  }

  /** test default shard key. */
  public void testGetShardKey() {
    final Properties p = new Properties();
    final StateStoreConfiguration config = new StateStoreConfiguration(p);
    assertEquals("01/", config.getShardKey(TEST_DIGEST));
  }
}
