/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.test;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.eclipse.smila.importing.DeltaImportStrategy;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.file.FileCrawlerService;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.workspace.WorkspaceHelper;

/** base class for tests of local and remote filecrawling. */
public abstract class AFileCrawlingTestBase extends AImportingIntegrationTest {

  /** waiting time for crawling files to be finished in milliseconds. */
  protected static final int WAIT_TIME_FOR_CRAWL_FILES_TO_FINISH = 60000;

  protected abstract String getCrawlJobName();

  public void testInitialCrawl() throws Exception {
    crawlFilesFromConfig("files10");
    assertTrue(_deltaService.getSourceIds().contains("files"));
    final int expectedRecordCount = 10;
    checkAddedBulks(expectedRecordCount);
    checkDeletedBulks(0);
  }

  public void testCrawlMultipleLevels() throws Exception {
    crawlFilesFromConfig("filesMultiLevel");
    assertTrue(_deltaService.getSourceIds().contains("files"));
    final int expectedRecordCount = 31;
    checkAddedBulks(expectedRecordCount);
    checkDeletedBulks(0);
  }

  public void testNoUpdatesCrawl() throws Exception {
    final File crawlDir = crawlFilesFromConfig("files10");
    checkAddedBulks(10);
    checkDeletedBulks(0);
    clearStore(STORENAME_BULKS);
    crawlFiles(crawlDir);
    checkAddedBulks(0);
    checkDeletedBulks(0);
  }

  public void testCrawlAdditionalFiles() throws Exception {
    crawlFilesFromConfig("files59");
    checkAddedBulks(59);
    checkDeletedBulks(0);
    clearStore(STORENAME_BULKS);
    crawlFilesFromConfig("files10");
    checkAddedBulks(10);
    checkDeletedBulks(0);
  }

  public void testCrawlUpdates() throws Exception {
    final File workspaceDir = copyConfigFilesToWorkspace("files10");
    crawlFiles(workspaceDir);
    checkAddedBulks(10);
    checkDeletedBulks(0);
    clearStore(STORENAME_BULKS);
    // modify half of the files
    final File[] files = workspaceDir.listFiles();
    for (int i = 0; i < files.length; i++) {
      if (i % 2 == 0) {
        files[i].setLastModified(System.currentTimeMillis());
      }
    }
    crawlFiles(workspaceDir);
    checkUpdateBulks(5);
    checkDeletedBulks(0);
  }

  public void testDeleteRemovedFiles() throws Exception {
    crawlFilesFromConfig("files10");
    checkAddedBulks(10);
    checkDeletedBulks(0);
    clearStore(STORENAME_BULKS);
    final File cleanDir = cleanCrawlDirectory();
    crawlFiles(cleanDir);
    checkAddedBulks(0);
    checkDeletedBulks(10);
  }

  public void testCrawlOtherFiles() throws Exception {
    crawlFilesFromConfig("files59");
    checkAddedBulks(59);
    checkDeletedBulks(0);
    clearStore(STORENAME_BULKS);
    cleanCrawlDirectory();
    crawlFilesFromConfig("files10");
    checkAddedBulks(10);
    checkDeletedBulks(59);
  }

  /** test import with delta strategy disabled. */
  public void testDeltaDisabled() throws Exception {
    crawlFilesFromConfig("files59", DeltaImportStrategy.DISABLED);
    checkAddedBulks(59);
    checkDeletedBulks(0);
    assertFalse(_deltaService.getSourceIds().contains("files"));
    clearStore(STORENAME_BULKS);
    crawlFilesFromConfig("files10", DeltaImportStrategy.DISABLED);
    checkAddedBulks(69);
    checkDeletedBulks(0);
    assertFalse(_deltaService.getSourceIds().contains("files"));
  }

  /** test that no records are deleted with delta strategy disabled. */
  public void testDeltaDisabledNoDelete() throws Exception {
    crawlFilesFromConfig("files59", DeltaImportStrategy.DISABLED);
    checkAddedBulks(59);
    checkDeletedBulks(0);
    assertFalse(_deltaService.getSourceIds().contains("files"));
    clearStore(STORENAME_BULKS);
    cleanCrawlDirectory();
    crawlFilesFromConfig("files10", DeltaImportStrategy.DISABLED);
    checkAddedBulks(10);
    checkDeletedBulks(0);
    assertFalse(_deltaService.getSourceIds().contains("files"));
  }

  /** test import with delta strategy for initial import. */
  public void testDeltaInitial() throws Exception {
    crawlFilesFromConfig("files59", DeltaImportStrategy.INITIAL);
    checkAddedBulks(59);
    checkDeletedBulks(0);
    assertEquals(59, _deltaService.countEntries("files", true));
    clearStore(STORENAME_BULKS);
    crawlFilesFromConfig("files10", DeltaImportStrategy.INITIAL);
    checkAddedBulks(69);
    checkDeletedBulks(0);
    assertEquals(69, _deltaService.countEntries("files", true));
  }

  /** test that no records are deleted with delta strategy for initial import. */
  public void testDeltaInitialNoDelete() throws Exception {
    crawlFilesFromConfig("files59", DeltaImportStrategy.INITIAL);
    checkAddedBulks(59);
    checkDeletedBulks(0);
    assertEquals(59, _deltaService.countEntries("files", true));
    cleanCrawlDirectory();
    clearStore(STORENAME_BULKS);
    crawlFilesFromConfig("files10", DeltaImportStrategy.INITIAL);
    checkAddedBulks(10);
    checkDeletedBulks(0);
    assertEquals(69, _deltaService.countEntries("files", true));
  }

  /** test import with delta strategy for additive import. */
  public void testDeltaAdditive() throws Exception {
    crawlFilesFromConfig("files59", DeltaImportStrategy.ADDITIVE);
    checkAddedBulks(59);
    checkDeletedBulks(0);
    assertEquals(59, _deltaService.countEntries("files", true));
    clearStore(STORENAME_BULKS);
    crawlFilesFromConfig("files10", DeltaImportStrategy.ADDITIVE);
    checkAddedBulks(10);
    checkDeletedBulks(0);
    assertEquals(69, _deltaService.countEntries("files", true));
  }

  /** test that no records are deleted with delta strategy for additive import. */
  public void testDeltaAdditiveNoDelete() throws Exception {
    crawlFilesFromConfig("files59", DeltaImportStrategy.ADDITIVE);
    checkAddedBulks(59);
    checkDeletedBulks(0);
    assertEquals(59, _deltaService.countEntries("files", true));
    cleanCrawlDirectory();
    clearStore(STORENAME_BULKS);
    crawlFilesFromConfig("files10", DeltaImportStrategy.ADDITIVE);
    checkAddedBulks(10);
    checkDeletedBulks(0);
    assertEquals(69, _deltaService.countEntries("files", true));
  }

  /** test import with delta strategy for full import. */
  public void testDeltaFull() throws Exception {
    crawlFilesFromConfig("files59", DeltaImportStrategy.FULL);
    checkAddedBulks(59);
    checkDeletedBulks(0);
    assertEquals(59, _deltaService.countEntries("files", true));
    clearStore(STORENAME_BULKS);
    crawlFilesFromConfig("files10", DeltaImportStrategy.FULL);
    checkAddedBulks(10);
    checkDeletedBulks(0);
    assertEquals(69, _deltaService.countEntries("files", true));
  }

  /** test that records are deleted with delta strategy for full import. */
  public void testDeltaFullDelete() throws Exception {
    crawlFilesFromConfig("files59", DeltaImportStrategy.FULL);
    checkAddedBulks(59);
    checkDeletedBulks(0);
    assertEquals(59, _deltaService.countEntries("files", true));
    cleanCrawlDirectory();
    clearStore(STORENAME_BULKS);
    crawlFilesFromConfig("files10", DeltaImportStrategy.FULL);
    checkAddedBulks(10);
    checkDeletedBulks(59);
    assertEquals(10, _deltaService.countEntries("files", true));
  }

  /** test crawling of compounds. */
  public void testCompoundCrawlingDeltaInitial() throws Exception {
    crawlFilesFromConfig("compounds", DeltaImportStrategy.INITIAL);
    checkAddedBulks(21); // 10 files in dir, 1 compound with 10 files.
    checkDeletedBulks(0);
    assertEquals(21, _deltaService.countEntries("files", true));
  }

  /** test crawling of compounds. */
  public void testCompoundCrawlingDeltaFull() throws Exception {
    final File crawlDir = crawlFilesFromConfig("compounds", DeltaImportStrategy.FULL);
    checkAddedBulks(21); // 10 files in dir, 1 compound with 10 files.
    checkDeletedBulks(0);
    assertEquals(21, _deltaService.countEntries("files", true));
    clearStore(STORENAME_BULKS);
    crawlFiles(crawlDir, DeltaImportStrategy.FULL);
    checkUpdateBulks(0); // nothing changed
    checkDeletedBulks(0);
    assertEquals(21, _deltaService.countEntries("files", true));
  }

  /** test crawling of updated compounds. */
  public void testCompoundCrawlingUpdateDeltaFull() throws Exception {
    final File crawlDir = crawlFilesFromConfig("compounds", DeltaImportStrategy.FULL);
    checkAddedBulks(21); // 10 files in dir, 1 compound with 10 files.
    checkDeletedBulks(0);
    assertEquals(21, _deltaService.countEntries("files", true));
    clearStore(STORENAME_BULKS);
    final File crawlCompound = new File(crawlDir, "files10.zip");
    final File updateCompound = ConfigUtils.getConfigFile(AllTests.BUNDLE_ID, "compounds-update/files10.zip");
    FileUtils.deleteQuietly(crawlCompound);
    FileUtils.copyFile(updateCompound, crawlCompound);
    FileUtils.touch(crawlCompound);
    crawlFiles(crawlDir, DeltaImportStrategy.FULL);
    checkAddOrUpdateBulks(6); // compound changed, 2 elements touched, 3 renamed -> 6 updates
    checkDeletedBulks(3); // 3 deletes.
    assertEquals(21, _deltaService.countEntries("files", true));
  }

  /** test crawling of compounds. */
  public void testCompoundCrawlingDeleteDeltaFull() throws Exception {
    final File crawlDir = crawlFilesFromConfig("compounds", DeltaImportStrategy.FULL);
    checkAddedBulks(21); // 10 files in dir, 1 compound with 10 files.
    checkDeletedBulks(0);
    assertEquals(21, _deltaService.countEntries("files", true));
    clearStore(STORENAME_BULKS);
    FileUtils.deleteQuietly(new File(crawlDir, "files10.zip"));
    crawlFiles(crawlDir, DeltaImportStrategy.FULL);
    checkUpdateBulks(0);
    checkDeletedBulks(11); // compound and 10 elements deleted.
    assertEquals(10, _deltaService.countEntries("files", true));
  }

  /** clears a store and waits for it to be really cleared... */
  protected void clearStore(final String storeName) throws ObjectStoreException, InterruptedException {
    int i = 10;
    boolean finished = false;
    do {
      _objectStore.clearStore(storeName);
      final Collection<StoreObject> storeObjectInfos = _objectStore.getStoreObjectInfos(storeName);
      if (storeObjectInfos == null || storeObjectInfos.isEmpty()) {
        finished = true;
      } else {
        Thread.sleep(100);
      }
    } while (i-- > 0 && !finished);
    assertTrue("Could not clean up object store " + storeName, finished);
  }

  protected String getCrawlDirectoryName() {
    return getCrawlJobName() + "-" + getName();
  }

  /** remove a workspace directory. */
  protected File cleanCrawlDirectory() throws IOException {
    final File workspaceDir = WorkspaceHelper.createWorkingDir(AllTests.BUNDLE_ID, getCrawlDirectoryName());
    FileUtils.cleanDirectory(workspaceDir);
    return workspaceDir;
  }

  /** copy a directory from configuration to workspace, skip .svn directories. */
  protected File copyConfigFilesToWorkspace(final String configDirName) throws IOException {
    final File configDir = ConfigUtils.getConfigFile(AllTests.BUNDLE_ID, configDirName);
    final File workspaceDir = WorkspaceHelper.createWorkingDir(AllTests.BUNDLE_ID, getCrawlDirectoryName());
    FileUtils
      .copyDirectory(configDir, workspaceDir, FileFilterUtils.makeSVNAware(FileFilterUtils.trueFileFilter()));
    return workspaceDir;
  }

  protected File crawlFilesFromConfig(final String configDirName) throws Exception {
    return crawlFilesFromConfig(configDirName, null);
  }

  protected File crawlFilesFromConfig(final String configDirName, final DeltaImportStrategy deltaUsage)
    throws Exception {
    final File workspaceDir = copyConfigFilesToWorkspace(configDirName);
    crawlFiles(workspaceDir, deltaUsage);
    return workspaceDir;
  }

  protected void crawlFiles(final File dirToCrawl) throws Exception {
    crawlFiles(dirToCrawl, null);
  }

  protected void crawlFiles(final File dirToCrawl, final DeltaImportStrategy deltaUsage) throws Exception {
    final String crawlJobId = startFileCrawlerJob(dirToCrawl, deltaUsage);
    try {
      waitForJobRunCompleted(getCrawlJobName(), crawlJobId, WAIT_TIME_FOR_CRAWL_FILES_TO_FINISH);
    } catch (final Error ex) {
      cancelJobQuietly(crawlJobId);
      throw ex;
    } catch (final Exception ex) {
      cancelJobQuietly(crawlJobId);
      throw ex;
    }
    _bulkbuilder.commitJob(JOBNAME_BUILDBULKS);
  }

  private void cancelJobQuietly(final String crawlJobId) {
    try {
      _jobRunEngine.cancelJob(getCrawlJobName(), crawlJobId);
    } catch (final Exception ex) {
      ex.printStackTrace();
    }
  }

  protected void checkAddOrUpdateBulks(final int expectedRecordCount) throws Exception {
    final Collection<StoreObject> bulks = _objectStore.getStoreObjectInfos(STORENAME_BULKS, BUCKET_ADDED);
    assertNotNull(bulks);
    if (expectedRecordCount == 0) {
      assertTrue(bulks.isEmpty());
    } else {
      assertEquals(expectedRecordCount, checkAddedRecords(bulks, null));
    }
  }

  protected void checkAddedBulks(final int expectedRecordCount) throws Exception {
    final Collection<StoreObject> bulks = _objectStore.getStoreObjectInfos(STORENAME_BULKS, BUCKET_ADDED);
    assertNotNull(bulks);
    if (expectedRecordCount == 0) {
      assertTrue(bulks.isEmpty());
    } else {
      assertEquals(expectedRecordCount, checkAddedRecords(bulks, false));
    }
  }

  protected void checkUpdateBulks(final int expectedRecordCount) throws ObjectStoreException, Exception {
    final Collection<StoreObject> bulks = _objectStore.getStoreObjectInfos(STORENAME_BULKS, BUCKET_ADDED);
    assertNotNull(bulks);
    assertEquals(expectedRecordCount, checkAddedRecords(bulks, true));
  }

  private int checkAddedRecords(final Collection<StoreObject> bulks, final Boolean update) throws Exception {
    int recordCount = 0;
    for (final StoreObject bulk : bulks) {
      final InputStream bulkStream = _objectStore.readObject(STORENAME_BULKS, bulk.getId());
      try (final BinaryObjectStreamIterator records = new BinaryObjectStreamIterator(bulkStream)) {
        while (records.hasNext()) {
          final Record record = records.next();
          assertNotNull(record);
          recordCount++;
          assertNotNull(record.getId());
          assertEquals("files", record.getSource());
          final AnyMap metadata = record.getMetadata();
          if (update != null) {
            if (update) {
              assertTrue(metadata.getBooleanValue(ImportingConstants.ATTRIBUTE_UPDATE));
            } else {
              assertFalse(metadata.containsKey(ImportingConstants.ATTRIBUTE_UPDATE));
            }
          }
          assertTrue(metadata.containsKey(ImportingConstants.ATTRIBUTE_DELTA_HASH));
          assertTrue(metadata.containsKey(FileCrawlerService.PROPERTY_FILE_NAME));
          assertTrue(metadata.containsKey(FileCrawlerService.PROPERTY_FILE_PATH));
          assertTrue(metadata.containsKey(FileCrawlerService.PROPERTY_FILE_FOLDER));
          assertTrue(metadata.get(FileCrawlerService.PROPERTY_FILE_LAST_MODIFIED).isDateTime());
          assertTrue(metadata.get(FileCrawlerService.PROPERTY_FILE_SIZE).isLong());
          if (metadata.containsKey(ImportingConstants.ATTRIBUTE_COMPOUNDFLAG)) {
            assertFalse(record.hasAttachment(FileCrawlerService.ATTACHMENT_FILE_CONTENT));
          } else {
            assertTrue(record.hasAttachment(FileCrawlerService.ATTACHMENT_FILE_CONTENT));
          }
        }
      } finally {
        IOUtils.closeQuietly(bulkStream);
      }
    }
    return recordCount;
  }

  protected void checkDeletedBulks(final int expectedRecordCount) throws Exception {
    final Collection<StoreObject> bulks = _objectStore.getStoreObjectInfos(STORENAME_BULKS, BUCKET_DELETED);
    assertNotNull(bulks);
    if (expectedRecordCount == 0) {
      assertTrue(bulks.isEmpty());
    } else {
      assertEquals(expectedRecordCount, checkDeletedRecords(bulks));
    }
  }

  private int checkDeletedRecords(final Collection<StoreObject> bulks) throws Exception {
    int recordCount = 0;
    for (final StoreObject bulk : bulks) {
      final InputStream bulkStream = _objectStore.readObject(STORENAME_BULKS, bulk.getId());
      try (final BinaryObjectStreamIterator records = new BinaryObjectStreamIterator(bulkStream)) {
        while (records.hasNext()) {
          final Record record = records.next();
          assertNotNull(record);
          recordCount++;
          assertNotNull(record.getId());
          assertEquals("files", record.getSource());
          assertEquals(2, record.getMetadata().size());
        }
      } finally {
        IOUtils.closeQuietly(bulkStream);
      }
    }
    return recordCount;
  }

  private String startFileCrawlerJob(final File dirToCrawl, final DeltaImportStrategy deltaUsage) throws Exception {
    final JobDefinition jobTemplate = _defPersistence.getJob(getCrawlJobName() + "Template");
    final AnyMap jobAny = DataFactory.DEFAULT.cloneAnyMap(jobTemplate.toAny(false)); // prevent changes in template
    jobAny.put("name", getCrawlJobName());
    jobAny.getMap("parameters").put("rootFolder", dirToCrawl.getAbsolutePath());
    if (deltaUsage != null) {
      jobAny.getMap("parameters").put("deltaImportStrategy", deltaUsage.getExternalName());
    }
    final JobDefinition job = new JobDefinition(jobAny);
    _defPersistence.addJob(job);
    return _jobRunEngine.startJob(getCrawlJobName(), JobRunMode.RUNONCE);
  }

}
