/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.ipc.bon;

import java.io.IOException;
import java.io.InputStream;

import org.eclipse.smila.ipc.IpcStreamReader;
import org.eclipse.smila.ipc.IpcToken;

/**
 * IPC reader implementation for the binary protocol (BON).
 * 
 * @author aweber
 */
public class BinaryStreamReader implements IpcStreamReader {

  /** current token. */
  private BinaryToken _currentToken;

  /** current value. */
  private byte[] _currentValue;

  /** underlying byte stream. */
  private final InputStream _stream;

  /**
   * @param stream
   *          BON byte stream.
   */
  BinaryStreamReader(final InputStream stream) {
    _stream = stream;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws IOException {
    if (_stream != null) {
      _stream.close();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void closeWithoutStream() throws IOException {
    // do nothing
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean currentBoolValue() {
    if (_currentToken == BinaryToken.SCALAR_BOOL_TRUE) {
      return true;
    } else if (_currentToken == BinaryToken.SCALAR_BOOL_FALSE) {
      return false;
    } else {
      throw new IllegalStateException("Current token is not a boolean value: " + _currentToken);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public double currentDoubleValue() {
    if (_currentToken.asIpcToken() != IpcToken.SCALAR_DOUBLE) {
      throw new IllegalStateException("Current token is not a double value: " + _currentToken);
    }
    return ValueTrafo.byte2double(_currentValue);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long currentLongValue() {
    if (_currentToken.asIpcToken() != IpcToken.SCALAR_INT) {
      throw new IllegalStateException("Current token is not an int value: " + _currentToken);
    }
    final long v = ValueTrafo.byte2long(_currentValue);
    return _currentToken.hasNegativeValue() ? -v : v;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String currentStringValue() {
    // conversion of binary values to strings is done to make a test case simpler...
    if (_currentToken.asIpcToken() == IpcToken.SCALAR_STRING || _currentToken.asIpcToken() == IpcToken.BINARY) {
      return ValueTrafo.byte2string(_currentValue);
    } else if (_currentToken.asIpcToken() == IpcToken.SCALAR_DOUBLE) {
      return String.valueOf(ValueTrafo.byte2double(_currentValue));
    } else if (_currentToken.asIpcToken() == IpcToken.SCALAR_INT) {
      return String.valueOf(ValueTrafo.byte2long(_currentValue));
    } else if (_currentToken == BinaryToken.SCALAR_BOOL_FALSE) {
      return String.valueOf(Boolean.FALSE);
    } else if (_currentToken == BinaryToken.SCALAR_BOOL_TRUE) {
      return String.valueOf(Boolean.TRUE);
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] currentBinaryValue() {
    return _currentValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean nextBoolValue() throws IOException {
    readTokenFromStream();
    return currentBoolValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public double nextDoubleValue() throws IOException {
    readTokenFromStream();
    readValueFromStream();
    return currentDoubleValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long nextLongValue() throws IOException {
    readTokenFromStream();
    readValueFromStream();
    return currentLongValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String nextStringValue() throws IOException {
    readTokenFromStream();
    readValueFromStream();
    return currentStringValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] nextBinaryValue() throws IOException {
    readTokenFromStream();
    readValueFromStream();
    return currentBinaryValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IpcToken nextToken() throws IOException {
    // read next token - should signal type and length of value
    readTokenFromStream();
    if (_currentToken == null) {
      return null;
    }
    // read value for scalar token types
    if (_currentToken.hasValue()) {
      readValueFromStream();
    } else {
      _currentValue = null;
    }
    return _currentToken.asIpcToken();
  }

  /**
   * read the next token from the stream.
   * 
   * @throws IOException
   *           reading the stream.
   */
  private void readTokenFromStream() throws IOException {
    _currentToken = readAndCheckToken();
    if (_currentToken == BinaryToken.CUSTOM) {
      _currentToken = readAndCheckToken();
      if (_currentToken.asIpcToken() != IpcToken.SCALAR_STRING) {
        throw new IllegalStateException("Invalid token after Custom token: " + _currentToken);
      }
      readValueFromStream();
      _currentToken = readAndCheckToken();
    }
  }

  private BinaryToken readAndCheckToken() throws IOException {
    final int b = _stream.read();
    if (b < 0) {
      return null;
    }
    final BinaryToken token = BinaryToken.getToken(b);
    if (token == null) {
      throw new IllegalStateException("Invalid token byte " + b);
    }
    return token;
  }

  /**
   * read the current value from the stream.
   * 
   * @throws IOException
   *           reading the stream.
   */
  private void readValueFromStream() throws IOException {
    readValueFromStream(_currentToken.valueLength());
    // Special handling for Strings:
    if (_currentToken.asIpcToken() == IpcToken.SCALAR_STRING) {
      // _currentValue contains only the length of the string - not the value.
      // So we have to read the value in a seperate step here:
      final int stringLength = ValueTrafo.byte2int(_currentValue);
      readValueFromStream(stringLength);
    }
    // Special handling for Binaries:
    if (_currentToken.asIpcToken() == IpcToken.BINARY) {
      // _currentValue contains only the length of the binary - not the value.
      // So we have to read the value in a seperate step here:
      final int binaryLength = ValueTrafo.byte2int(_currentValue);
      readValueFromStream(binaryLength);
    }
  }

  /**
   * read required value length from stream.
   * 
   * @param length
   *          required value length
   * @throws IOException
   *           error reading stream
   */
  private void readValueFromStream(final int length) throws IOException {
    _currentValue = new byte[length];
    int readLength = 0;
    while (readLength < length) {
      final int nextRead = _stream.read(_currentValue, readLength, length - readLength);
      if (nextRead < 0) {
        throw new IOException("got only " + readLength + " value bytes when " + length + " where required.");
      }
      readLength += nextRead;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getStream() {
    return _stream;
  }

}
