/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH.  
 * All rights reserved. This program and the accompanying materials are made available
 * under the terms of the Eclipse Public License v1.0 which accompanies this distribution, 
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.ipc.bon;

import java.io.IOException;
import java.io.OutputStream;

import org.eclipse.smila.ipc.IpcStreamWriter;

/**
 * IPC writer implementation for the binary protocol (BON).
 * 
 * @author aweber
 */
public class BinaryStreamWriter implements IpcStreamWriter {

  /** target stream. */
  private final OutputStream _stream;

  /**
   * @param stream
   *          target stream.
   */
  BinaryStreamWriter(final OutputStream stream) {
    _stream = stream;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws IOException {
    if (_stream != null) {
      _stream.close();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void closeWithoutStream() throws IOException {
    // do nothing
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeObjectEnd() throws IOException {
    _stream.write(BinaryToken.OBJECT_END.byteValue());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeObjectStart() throws IOException {
    _stream.write(BinaryToken.OBJECT_START.byteValue());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeMappingEnd() throws IOException {
    _stream.write(BinaryToken.MAPPING_END.byteValue());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeMappingStart() throws IOException {
    _stream.write(BinaryToken.MAPPING_START.byteValue());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeScalarBoolean(final boolean b) throws IOException {
    if (b) {
      _stream.write(BinaryToken.SCALAR_BOOL_TRUE.byteValue());
    } else {
      _stream.write(BinaryToken.SCALAR_BOOL_FALSE.byteValue());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeScalarDouble(final double d) throws IOException {
    _stream.write(BinaryToken.SCALAR_DOUBLE.byteValue());
    _stream.write(ValueTrafo.double2byte(d)); // double -> 8 bytes length    
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeScalarInt(final int i) throws IOException {
    writeScalarLong(i);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeScalarLong(long i) throws IOException {
    boolean neg = false;
    if (i < 0) {
      neg = true;
      i = -i; // convert to positive value
    }
    final int byteLength = ValueTrafo.getBytesLength(i);
    final byte[] b = ValueTrafo.long2byte(i, byteLength);

    switch (byteLength) {
      case 1:
        _stream.write(neg ? BinaryToken.SCALAR_INT_1_N.byteValue() : BinaryToken.SCALAR_INT_1.byteValue());
        break;
      case 2:
        _stream.write(neg ? BinaryToken.SCALAR_INT_2_N.byteValue() : BinaryToken.SCALAR_INT_2.byteValue());
        break;
      case 3:
        _stream.write(neg ? BinaryToken.SCALAR_INT_3_N.byteValue() : BinaryToken.SCALAR_INT_3.byteValue());
        break;
      case 4:
        _stream.write(neg ? BinaryToken.SCALAR_INT_4_N.byteValue() : BinaryToken.SCALAR_INT_4.byteValue());
        break;
      case 5:
        _stream.write(neg ? BinaryToken.SCALAR_INT_5_N.byteValue() : BinaryToken.SCALAR_INT_5.byteValue());
        break;
      case 6:
        _stream.write(neg ? BinaryToken.SCALAR_INT_6_N.byteValue() : BinaryToken.SCALAR_INT_6.byteValue());
        break;
      case 7:
        _stream.write(neg ? BinaryToken.SCALAR_INT_7_N.byteValue() : BinaryToken.SCALAR_INT_7.byteValue());
        break;
      case 8:
        _stream.write(neg ? BinaryToken.SCALAR_INT_8_N.byteValue() : BinaryToken.SCALAR_INT_8.byteValue());
        break;
      default:
        throw new IllegalStateException("invalid byte size");
    }
    _stream.write(b);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeScalarString(final String s) throws IOException {
    final byte[] b = ValueTrafo.string2byte(s);
    final byte[] bLength = ValueTrafo.long2byte(b.length); // byte array containing string length info
    if (bLength.length > 4) {
      throw new IllegalArgumentException("string too long, byte length: " + bLength.length);
    } else if (bLength.length == 1) {
      _stream.write(BinaryToken.SCALAR_STRING_1.byteValue());
    } else if (bLength.length == 2) {
      _stream.write(BinaryToken.SCALAR_STRING_2.byteValue());
    } else if (bLength.length == 3) {
      _stream.write(BinaryToken.SCALAR_STRING_3.byteValue());
    } else if (bLength.length == 4) {
      _stream.write(BinaryToken.SCALAR_STRING_4.byteValue());
    }
    _stream.write(bLength);
    _stream.write(b);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeBinary(final byte[] b) throws IOException {
    final byte[] bLength = ValueTrafo.long2byte(b.length); // byte array containing binary length info
    if (bLength.length > 4) {
      throw new IllegalArgumentException("binary too long, byte length: " + bLength.length);
    } else if (bLength.length == 1) {
      _stream.write(BinaryToken.BINARY_1.byteValue());
    } else if (bLength.length == 2) {
      _stream.write(BinaryToken.BINARY_2.byteValue());
    } else if (bLength.length == 3) {
      _stream.write(BinaryToken.BINARY_3.byteValue());
    } else if (bLength.length == 4) {
      _stream.write(BinaryToken.BINARY_4.byteValue());
    }
    _stream.write(bLength);

    // split into 2 writes to improve internal memory management for large objects in ByteArrayOutputStream 
    final int halfsize = b.length / 2;
    _stream.write(b, 0, halfsize);
    _stream.write(b, halfsize, b.length - halfsize);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeSequenceEnd() throws IOException {
    _stream.write(BinaryToken.SEQUENCE_END.byteValue());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeSequenceStart() throws IOException {
    _stream.write(BinaryToken.SEQUENCE_START.byteValue());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeMappingKey(final String s) throws IOException {
    writeScalarString(s);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeAttachmentsStart() throws IOException {
    _stream.write(BinaryToken.ATTACHMENTS_START.byteValue());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeAttachmentsEnd() throws IOException {
    _stream.write(BinaryToken.ATTACHMENTS_END.byteValue());
  }

  @Override
  public boolean hasBinarySupport() {
    return true;
  }

}
