/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.ipc.bon;

import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.smila.ipc.IpcToken;

/**
 * Tokens used in the binary protocol implementation. Contains mapping between binary tokens and standard IPC Tokens.
 * See <a href=
 * "http://wiki.eclipse.org/SMILA/Documentation/Data_Model_and_Serialization_Formats#BON_Binary_Object_Notation_Format">
 * http://wiki.eclipse.org/SMILA/Documentation/Data_Model_and_Serialization_Formats#BON_Binary_Object_Notation_Format
 * </a> for specification.
 * 
 * @author aweber
 */
public enum BinaryToken {
  /** (followed by) 1 byte positive integer. */
  SCALAR_INT_1(IpcToken.SCALAR_INT, 0, 1, false),
  /** (followed by) 2 byte positive integer. */
  SCALAR_INT_2(IpcToken.SCALAR_INT, 1, 2, false),
  /** (followed by) 3 byte positive integer. */
  SCALAR_INT_3(IpcToken.SCALAR_INT, 2, 3, false),
  /** (followed by) 4 byte positive integer. */
  SCALAR_INT_4(IpcToken.SCALAR_INT, 3, 4, false),
  /** (followed by) 5 byte positive integer. */
  SCALAR_INT_5(IpcToken.SCALAR_INT, 4, 5, false),
  /** (followed by) 6 byte positive integer. */
  SCALAR_INT_6(IpcToken.SCALAR_INT, 5, 6, false),
  /** (followed by) 7 byte positive integer. */
  SCALAR_INT_7(IpcToken.SCALAR_INT, 6, 7, false),
  /** (followed by) 8 byte positive integer. */
  SCALAR_INT_8(IpcToken.SCALAR_INT, 7, 8, false),
  /** (followed by) 1 byte negative integer. */
  SCALAR_INT_1_N(IpcToken.SCALAR_INT, 8, 1, true),
  /** (followed by) 2 byte negative integer. */
  SCALAR_INT_2_N(IpcToken.SCALAR_INT, 9, 2, true),
  /** (followed by) 3 byte negative integer. */
  SCALAR_INT_3_N(IpcToken.SCALAR_INT, 10, 3, true),
  /** (followed by) 4 byte negative integer. */
  SCALAR_INT_4_N(IpcToken.SCALAR_INT, 11, 4, true),
  /** (followed by) 5 byte negative integer. */
  SCALAR_INT_5_N(IpcToken.SCALAR_INT, 12, 5, true),
  /** (followed by) 6 byte negative integer. */
  SCALAR_INT_6_N(IpcToken.SCALAR_INT, 13, 6, true),
  /** (followed by) 7 byte negative integer. */
  SCALAR_INT_7_N(IpcToken.SCALAR_INT, 14, 7, true),
  /** (followed by) 8 byte negative integer. */
  SCALAR_INT_8_N(IpcToken.SCALAR_INT, 15, 8, true),
  /** boolean TRUE. */
  SCALAR_BOOL_TRUE(IpcToken.SCALAR_BOOL, 16, 0, false),
  /** boolean FALSE. */
  SCALAR_BOOL_FALSE(IpcToken.SCALAR_BOOL, 17, 0, false),
  /** (followed by) 8 byte float value. */
  SCALAR_DOUBLE(IpcToken.SCALAR_DOUBLE, 19, 8, false),
  /** (followed by) 1 byte string length (and string). */
  SCALAR_STRING_1(IpcToken.SCALAR_STRING, 21, 1, false),
  /** (followed by) 2 byte string length (and string). */
  SCALAR_STRING_2(IpcToken.SCALAR_STRING, 22, 2, false),
  /** (followed by) 3 byte string length (and string). */
  SCALAR_STRING_3(IpcToken.SCALAR_STRING, 23, 3, false),
  /** (followed by) 4 byte string length (and string). */
  SCALAR_STRING_4(IpcToken.SCALAR_STRING, 24, 4, false),
  /** object start. */
  OBJECT_START(IpcToken.OBJECT_START, 25, 0, false),
  /** object end. */
  OBJECT_END(IpcToken.OBJECT_END, 28, 0, false),
  /** sequence start. */
  SEQUENCE_START(IpcToken.SEQUENCE_START, 29, 0, false),
  /** sequence end. */
  SEQUENCE_END(IpcToken.SEQUENCE_END, 30, 0, false),
  /** mapping start. */
  MAPPING_START(IpcToken.MAPPING_START, 31, 0, false),
  /** mapping end. */
  MAPPING_END(IpcToken.MAPPING_END, 32, 0, false),
  /** attachment list start. */
  ATTACHMENTS_START(IpcToken.ATTACHMENTS_START, 33, 0, false),
  /** attachment list end. */
  ATTACHMENTS_END(IpcToken.ATTACHMENTS_END, 34, 0, false),
  /** (followed by) 1-byte binary object length (and binary object). */
  BINARY_1(IpcToken.BINARY, 35, 1, false),
  /** (followed by) 2-byte binary object length (and binary object). */
  BINARY_2(IpcToken.BINARY, 36, 2, false),
  /** (followed by) 3-byte binary object length (and binary object). */
  BINARY_3(IpcToken.BINARY, 37, 3, false),
  /** (followed by) 4-byte binary object length (and binary object). */
  BINARY_4(IpcToken.BINARY, 38, 4, false),
  /** custom type token, followed by string to name the custom type. Currently ignored in SMILA. */
  CUSTOM(null, 43, 4, false);

  /** mapping of byte values to token instances. */
  private static final Map<Byte, BinaryToken> REVERSE_LOOKUP = new HashMap<Byte, BinaryToken>(35);

  /** IpcToken name belonging to this token instance. */
  private final IpcToken _ipcToken;

  /** byte value in serialization for this token. */
  private final byte _byteValue;

  /** length in bytes of associated token value. */
  private final int _valueLength;

  /** true if the token value is a negative number. */
  private final boolean _negativeValue;

  static {
    // initialize reverse lookup map with all enum values
    for (final BinaryToken t : EnumSet.allOf(BinaryToken.class)) {
      REVERSE_LOOKUP.put(t.byteValue(), t);
    }
  }

  /**
   * create new token with the given specification.
   * 
   * @param ipcToken
   *          IpcToken name.
   * @param value
   *          byte value.
   * @param length
   *          length of value, if token has a value.
   * @param negative
   *          true if this token represents a negative number.
   */
  BinaryToken(final IpcToken ipcToken, final int value, final int length, final boolean negative) {
    this._byteValue = (byte) value;
    this._valueLength = length;
    this._ipcToken = ipcToken;
    this._negativeValue = negative;
  }

  /**
   * @return byte representation of token.
   */
  public byte byteValue() {
    return _byteValue;
  }

  /**
   * @return byte length of token value.
   */
  public int valueLength() {
    return _valueLength;
  }

  /**
   * @return IpcToken representation of binary token.
   */
  public IpcToken asIpcToken() {
    return _ipcToken;
  }

  /**
   * @param byteValue
   *          a byte value.
   * @return token for given byte value.
   */
  public static BinaryToken getToken(final int byteValue) {
    return REVERSE_LOOKUP.get((byte) byteValue);
  }

  /**
   * @return <code>true</code> if token has value, otherwise <code>false</code>.
   */
  public boolean hasValue() {
    return _valueLength > 0;
  }

  /**
   * @return <code>true</code> if token has negative value, otherwise <code>false</code>.
   * 
   *         This is needed for scalar integers.
   */
  public boolean hasNegativeValue() {
    return _negativeValue;
  }

}
