/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.jdbc.test;

import java.io.UnsupportedEncodingException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.sql.rowset.serial.SerialBlob;
import javax.sql.rowset.serial.SerialClob;

import org.eclipse.smila.common.logging.MessageCollector;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.jdbc.JdbcProvider;
import org.eclipse.smila.jdbc.internal.ResultSetToAnyConverter;

public class TestResultSetToAnyConverter extends JdbcTestBase {

  public void testResultSetToAnyConverter() throws Exception {
    final Connection conn = _jdbcProvider.getConnection(getConnectUrl(), getProperties());
    final Statement s = conn.createStatement();
    // check that the database is really created.
    final ResultSet resultSet = s.executeQuery("select * from " + DB_TABLE_NAME + " order by id asc");
    final Collection<Record> records =
      ResultSetToAnyConverter.convertResultSetToAny(resultSet, Long.MAX_VALUE, new MessageCollector.Ignore());
    assertEquals(getNumberOfRowsToCreate(), records.size());

    final List<Record> recordsAsList = new ArrayList<Record>(records);
    final int maxNo = records.size() - 1;

    // check first record
    Record recordToCheck = recordsAsList.get(0);
    AnyMap metadata = recordToCheck.getMetadata();
    assertEquals(1, metadata.size());
    assertTrue(metadata.containsKey("ID"));
    assertEquals(0, recordToCheck.attachmentSize());

    // check last record...
    recordToCheck = recordsAsList.get(maxNo);
    metadata = recordToCheck.getMetadata();
    checkAttributes(metadata, DataFactory.DEFAULT.createLongValue(Long.valueOf(maxNo)), "smallint_val", "int_val",
      "bigint_val");
    checkAttributes(metadata, DataFactory.DEFAULT.createDoubleValue(Double.valueOf(maxNo * 1.1)), "decimal_val",
      "numeric_val", "double_val", "float_val", "real_val");
    checkAttributes(metadata, DataFactory.DEFAULT.createStringValue(String.valueOf(maxNo)), "char_val",
      "varchar_val");
    final Date time = metadata.getDateTimeValue("TIME_VAL");
    final Calendar cal = Calendar.getInstance();
    cal.setTime(time);
    assertEquals(13, cal.get(Calendar.HOUR_OF_DAY));
    assertEquals(13, cal.get(Calendar.MINUTE));
    assertEquals(13, cal.get(Calendar.SECOND));
    final Date datetime = metadata.getDateTimeValue("TIMESTAMP_VAL");
    cal.setTime(datetime);
    assertEquals(2000, cal.get(Calendar.YEAR));
    assertEquals(1, cal.get(Calendar.MONTH));
    assertEquals(1, cal.get(Calendar.DAY_OF_MONTH));
    assertEquals(13, cal.get(Calendar.HOUR_OF_DAY));
    assertEquals(13, cal.get(Calendar.MINUTE));
    assertEquals(13, cal.get(Calendar.SECOND));
    final Date date = metadata.getDateTimeValue("DATE_VAL");
    cal.setTime(date);
    assertEquals(2000, cal.get(Calendar.YEAR));
    assertEquals(1, cal.get(Calendar.MONTH));
    assertEquals(1, cal.get(Calendar.DAY_OF_MONTH));
    assertEquals(0, cal.get(Calendar.HOUR_OF_DAY));
    assertEquals(0, cal.get(Calendar.MINUTE));
    assertEquals(0, cal.get(Calendar.SECOND));

    final String expectedString = String.valueOf(maxNo);
    assertTrue(recordToCheck.hasAttachment("VARBINARY_VAL"));
    String resultString = new String(recordToCheck.getAttachmentAsBytes("VARBINARY_VAL"), "utf-8");
    assertEquals(expectedString, resultString);
    assertTrue(recordToCheck.hasAttachment("CLOB_VAL"));
    resultString = new String(recordToCheck.getAttachmentAsBytes("CLOB_VAL"), "utf-8");
    assertEquals(expectedString + "äöü", resultString);
    assertTrue(recordToCheck.hasAttachment("BLOB_VAL"));
    resultString = new String(recordToCheck.getAttachmentAsBytes("BLOB_VAL"), "utf-8");
    assertEquals(expectedString, resultString);
  }

  public void testResultSetToAnyConverterMaxAttachmentSize() throws Exception {
    final Connection conn = _jdbcProvider.getConnection(getConnectUrl(), getProperties());
    final Statement s = conn.createStatement();
    // check that the database is really created.
    final ResultSet resultSet = s.executeQuery("select * from " + DB_TABLE_NAME + " order by id asc");
    final List<String> messages = new ArrayList<>();
    final Collection<Record> records =
      ResultSetToAnyConverter.convertResultSetToAny(resultSet, 0, new MessageCollector() {
        @Override
        public void add(final String message) {
          messages.add(message);
        }
      });
    assertEquals(getNumberOfRowsToCreate(), records.size());

    final List<Record> recordsAsList = new ArrayList<Record>(records);
    final int maxNo = records.size() - 1;

    assertEquals(maxNo * 3, messages.size()); // each but the first record has three attachment values.
    // check first record
    for (final Record record : recordsAsList) {
      assertFalse(record.hasAttachments());
    }
  }

  /** creates the table and stores some dummy data in it. */
  @Override
  protected void createTestDatabase(final JdbcProvider jdbcProvider) throws SQLException,
    UnsupportedEncodingException {
    final Connection conn = jdbcProvider.getConnection(getConnectUrl(), getProperties());
    final Statement s = conn.createStatement();
    try {
      s.execute("drop table " + DB_TABLE_NAME);
    } catch (final SQLException e) {
      ; // ignore
    }
    s.execute("create table " + DB_TABLE_NAME + "(smallint_val smallint," + // 1 long
      " int_val int," + // 2 long
      " bigint_val bigint," + // 3 long
      " decimal_val decimal(10,2)," + // 4 double
      " numeric_val numeric(10,2)," + // 5 double
      " double_val double," + // 6 double
      " float_val float," + // 7 double
      " real_val real," + // 8 double
      " char_val char(40)," + // 9 string
      " varchar_val varchar(40)," + // 10 string
      " varbinary_val varchar(40) for bit data," + // 11 byte[]
      " time_val time," + // 12 datetime
      " timestamp_val timestamp," + // 13 datetime
      " date_val date," + // 14 date
      " blob_val blob," + // 15 binary att
      " clob_val clob," + // 16 binary att
      " id int)"); // 17 id for select ordering

    final Calendar cal = Calendar.getInstance();
    cal.set(2000, 01, 01, 13, 13, 13);
    final Date time = cal.getTime();
    final PreparedStatement psInsert =
      conn.prepareStatement("insert into " + DB_TABLE_NAME + " values (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
    for (int i = 0; i < getNumberOfRowsToCreate(); i++) {
      if (i == 0) {
        psInsert.setNull(1, Types.SMALLINT);
        psInsert.setNull(2, Types.INTEGER);
        psInsert.setNull(3, Types.BIGINT);
        psInsert.setNull(4, Types.DECIMAL);
        psInsert.setNull(5, Types.NUMERIC);
        psInsert.setNull(6, Types.DOUBLE);
        psInsert.setNull(7, Types.FLOAT);
        psInsert.setNull(8, Types.REAL);
        psInsert.setNull(9, Types.CHAR);
        psInsert.setNull(10, Types.VARCHAR);
        psInsert.setNull(11, Types.VARBINARY);
        psInsert.setNull(12, Types.TIME);
        psInsert.setNull(13, Types.TIMESTAMP);
        psInsert.setNull(14, Types.DATE);
        psInsert.setNull(15, Types.BLOB);
        psInsert.setNull(16, Types.CLOB);
      } else {
        psInsert.setInt(1, i);
        psInsert.setInt(2, i);
        psInsert.setLong(3, i);
        psInsert.setDouble(4, i * 1.1);
        psInsert.setDouble(5, i * 1.1);
        psInsert.setDouble(6, i * 1.1);
        psInsert.setDouble(7, i * 1.1);
        psInsert.setDouble(8, i * 1.1);
        psInsert.setString(9, String.valueOf(i));
        psInsert.setString(10, String.valueOf(i));
        psInsert.setBytes(11, String.valueOf(i).getBytes("utf-8"));
        psInsert.setTime(12, new Time(time.getTime()));
        psInsert.setTimestamp(13, new Timestamp(time.getTime()));
        psInsert.setDate(14, new java.sql.Date(time.getTime()));
        psInsert.setBlob(15, new SerialBlob(String.valueOf(i).getBytes("utf-8")));
        final String testString = String.valueOf(i) + "äöü";
        final char[] chars = new char[testString.length()];
        testString.getChars(0, testString.length(), chars, 0);
        psInsert.setClob(16, new SerialClob(chars));
      }
      psInsert.setInt(17, i);
      psInsert.execute();
    }
  }

  /** @return the number of rows to create. */
  @Override
  protected int getNumberOfRowsToCreate() {
    return 10;
  }

  private void checkAttributes(final AnyMap metadata, final Value value, final String... attributeNames) {
    for (final String attributeName : attributeNames) {
      final String normalizedAttributeName = attributeName.toUpperCase(Locale.ENGLISH);
      if (value.isDouble()) {
        final double doubleValue = value.asDouble().doubleValue();
        assertTrue(normalizedAttributeName + " had wrong value.",
          doubleValue - 0.1 < metadata.getDoubleValue(normalizedAttributeName).doubleValue()
            && metadata.getDoubleValue(normalizedAttributeName).doubleValue() < doubleValue + 0.2);
      } else if (value.isString()) {
        final String stringValueTrimmed = value.asString().trim();
        final String convertedValueTrimmed = metadata.getStringValue(normalizedAttributeName).trim();
        assertEquals(normalizedAttributeName + " had wrong value.", stringValueTrimmed, convertedValueTrimmed);
      } else {
        assertEquals(normalizedAttributeName + " had wrong value.", value, metadata.get(normalizedAttributeName));
      }
    }
  }
}
