/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.jdbc;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.smila.common.logging.MessageCollector;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.util.RecordMerger;
import org.eclipse.smila.jdbc.internal.ResultSetToAnyConverter;

/**
 * Class for executing multiple SQL requests by using one Connection and one PreparedStatement. Must be closed after
 * processing to clean up DB resources!
 */
public class SqlExecutor implements AutoCloseable {
  private final String _sql;

  private final Connection _con;

  private final PreparedStatement _stmt;

  private final long _maxAttachmentSize;

  private final MessageCollector _messages;

  /**
   * @param con
   *          a database connection
   * @param sql
   *          an SQL statement which will be used as PreparedStatement, so it may contain '?' parameters.
   */
  public SqlExecutor(final Connection con, final String sql, final long maxAttachmentSize,
    final MessageCollector messages) throws SQLException {
    _sql = sql;
    _con = con;
    _stmt = con.prepareStatement(sql);
    _maxAttachmentSize = maxAttachmentSize;
    _messages = messages;
  }

  public String getSql() {
    return _sql;
  }

  /**
   * @param parameters
   *          the parameters for the SQL (Prepared)statement given in the constructor. Values must not be null.
   * @return one record. If SQL result contains multiple results, they are merged in one record.
   */
  public Record executeAndMerge(final List<Value> parameters) throws SQLException, IOException {
    final Collection<Record> result = execute(parameters);
    // if we have more than one result -> merge result to one record
    return mergeRecords(result);
  }

  /**
   * @param parameters
   *          the parameters for the SQL (Prepared)statement given in the constructor. Values must not be null.
   * @return A collection of records.
   */
  public Collection<Record> execute(final List<Value> parameters) throws SQLException, IOException {
    ResultSet rs = null;
    setParameters(parameters);
    rs = _stmt.executeQuery();
    return ResultSetToAnyConverter.convertResultSetToAny(rs, _maxAttachmentSize, _messages);
  }

  /**
   * @param parameters
   *          the parameters for the SQL (Prepared)statement given in the constructor. Values must not be null.
   * @return number of updated rows
   */
  public int executeUpdate(final List<Value> parameters) throws SQLException, IOException {
    setParameters(parameters);
    return _stmt.executeUpdate();
  }

  /**
   * execute update, supports null parameter values, too.
   * 
   * @param parameters
   *          the parameters for the SQL (Prepared)statement given in the constructor.
   * @param nullType
   *          SQL type for null values in parameters, see {@link java.sql.Types}.
   * @return number of updated rows
   */
  public int executeUpdate(final List<Value> parameters, final int nullType) throws SQLException, IOException {
    setParameters(parameters, nullType);
    return _stmt.executeUpdate();
  }

  private void setParameters(final List<Value> parameters) throws SQLException {
    // PreparedStatement uses list of parameters, parameterIndex starts with 1
    for (int j = 0; j < parameters.size(); j++) {
      final Any param = parameters.get(j);
      if (param == null) {
        throw new IllegalArgumentException("NULL parameters are not supported.");
      }
      _stmt.setObject(j + 1, param.asValue().getObject());
    }
  }

  private void setParameters(final List<Value> parameters, final int nullType) throws SQLException {
    // PreparedStatement uses list of parameters, parameterIndex starts with 1
    for (int j = 0; j < parameters.size(); j++) {
      final Any param = parameters.get(j);
      if (param == null) {
        _stmt.setNull(j + 1, nullType);
      } else {
        _stmt.setObject(j + 1, param.asValue().getObject());
      }
    }
  }

  /** @return merged record for given records to merge. */
  private Record mergeRecords(final Collection<Record> recordsToMerge) {
    if (recordsToMerge.size() == 0) {
      return null;
    }
    if (recordsToMerge.size() == 1) {
      return recordsToMerge.iterator().next();
    }
    final Iterator<Record> it = recordsToMerge.iterator();
    final Record result = it.next();
    while (it.hasNext()) {
      final Record r = it.next();
      RecordMerger.mergeRecords(result, r, true, true);
    }
    return result;
  }

  /** close statement and db connection. */
  @Override
  public void close() {
    if (_stmt != null) {
      try {
        _stmt.close();
      } catch (final Exception e) {
      }
    }
    if (_con != null) {
      try {
        _con.close();
      } catch (final Exception e) {
      }
    }
  }

  @Override
  /** just another chance if close() was not called. (double close() on Statement and Connection is ignored. */
  protected void finalize() throws Throwable {
    close();
  }
}
