/*******************************************************************************
 * Copyright (c) 2008, 2012 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.jdbc.internal;

import static java.lang.String.format;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Enumeration;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.jdbc.JdbcProvider;
import org.osgi.framework.Bundle;
import org.osgi.service.component.ComponentContext;

/**
 * default implementation of the {@link JdbcProvider} interface. The service assumes that JDBC driver JARs are either
 * added to the Bundle-ClassPath of this bundle, or the necessary packages are added to the Imported-Packages. In the
 * latter case the class names of the drivers must be listed in the "SMILA-JDBC" manifest header so that they can be
 * loaded by the service during activation. If the driver JARs are added to the Bundle-ClassPath and are
 * JDBC-4-compliant, the driver classes should be auto-detected from the contained
 * <tt>META-INF/services/java.sql.Driver</tt> files. If this does not work, you can still list the class name in the
 * "SMILA-JDBC" header for these drivers, too.
 */
public class JdbcProviderImpl implements JdbcProvider {

  private final Log _log = LogFactory.getLog(getClass());

  @Override
  public Connection getConnection(final String url, final String user, final String password) throws SQLException {
    return DriverManager.getConnection(url, user, password);
  }

  @Override
  public Connection getConnection(final String url, final Properties properties) throws SQLException {
    return DriverManager.getConnection(url, properties);
  }

  /**
   * initialize {@link DriverManager} to auto-detect driver classes from the Bundle-ClassPath, and add driver classes
   * listed in the SMILA-JDBC manifest header.
   */
  protected void activate(final ComponentContext context) {
    final Bundle bundle = context.getBundleContext().getBundle();
    // load drivers from META-INF/services files of jars in bundle.
    _log.info("These JDBC drivers were detected automagically:");
    logDriverInfo();
    _log.info("Loading additional JDBC drivers specified in manifest header:");
    final String driverHeader = bundle.getHeaders().get("SMILA-JDBC");
    final String[] driverNames = driverHeader.split(",");
    for (final String driverName : driverNames) {
      try {
        @SuppressWarnings("unchecked")
        final Class<Driver> driverClass = (Class<Driver>) Class.forName(driverName);
        final Driver driver = driverClass.newInstance();
        logDriverInfo(driver);
      } catch (final Exception ex) {
        _log.warn("JDBC driver class '" + driverName + "' could not be loaded.", ex);
      }
    }
  }

  /** write all current JDBC driver class names and versions to the log. */
  private void logDriverInfo() {
    final Enumeration<Driver> knownDrivers = DriverManager.getDrivers();
    while (knownDrivers.hasMoreElements()) {
      final Driver driver = knownDrivers.nextElement();
      logDriverInfo(driver);
    }
  }

  /** write class names and version of given to the log. */
  private void logDriverInfo(final Driver driver) {
    _log.info(format("* %s %d.%d", driver.getClass(), driver.getMajorVersion(), driver.getMinorVersion()));
  }
}
