/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.persistence.objectstore.test;

import java.util.Collection;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.persistence.PermanentStorage;
import org.eclipse.smila.jobmanager.persistence.objectstore.PermanentStorageObjectstore;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Test class with tests for the definition storage.
 * 
 */
public class TestPermanentStorage extends DeclarativeServiceTestCase {

  /** The number of test runs. */
  private static final int NUMBER_OF_RUNS = 100;

  /**
   * Test if HttpServiceImpl service was successfully started and registered.
   * 
   * @throws Exception
   *           no service found.
   */
  public void testService() throws Exception {
    final PermanentStorage service = getService(PermanentStorage.class);
    assertNotNull(service);
    assertTrue(service instanceof PermanentStorageObjectstore);
  }

  /**
   * Test if a bucket can be created, stored and removed from the definition storage and if the number of retrieved
   * bucket names matches the number of buckets in the store.
   * 
   * @throws Exception
   *           any exception that occurred during this test (e.g. no service found).
   */
  public void testBuckets() throws Exception {
    final PermanentStorage service = getService(PermanentStorage.class);
    for (int i = 0; i < NUMBER_OF_RUNS; i++) {
      final AnyMap any = DataFactory.DEFAULT.createAnyMap();
      final String name = "myName" + i;
      any.put("name", name);
      any.put("type", "recordBulks");
      final BucketDefinition bucket = new BucketDefinition(any);
      service.addBucket(bucket);
      final Collection<String> names = service.getBuckets();
      assertEquals(1, names.size());
      assertEquals(name, names.iterator().next());
      final BucketDefinition read = service.getBucket(name);
      assertNotNull(read);
      service.removeBucket(name);
      final Collection<String> noNames = service.getBuckets();
      assertTrue(noNames.isEmpty());
    }
  }

  /**
   * Test if a workflow can be created, stored and removed from the definition storage and if the number of retrieved
   * workflow names matches the number of workflows in the store.
   * 
   * @throws Exception
   *           any exception that occurred during this test (e.g. no service found).
   */
  public void testWorkflows() throws Exception {
    final PermanentStorage service = getService(PermanentStorage.class);
    final AnyMap startAction = DataFactory.DEFAULT.createAnyMap();
    startAction.put("worker", "startWorker");
    for (int i = 0; i < NUMBER_OF_RUNS; i++) {
      final AnyMap any = DataFactory.DEFAULT.createAnyMap();
      final String name = "myName" + i;
      any.put("name", name);
      any.put("startAction", startAction);
      final WorkflowDefinition workflow = new WorkflowDefinition(any);
      service.addWorkflow(workflow);
      final Collection<String> names = service.getWorkflows();
      assertEquals(1, names.size());
      assertEquals(name, names.iterator().next());
      final WorkflowDefinition read = service.getWorkflow(name);
      assertNotNull(read);
      service.removeWorkflow(name);
      final Collection<String> noNames = service.getWorkflows();
      assertTrue(noNames.isEmpty());
    }
  }

  /**
   * Test if a job can be created, stored and removed from the definition storage and if the number of retrieved job
   * names matches the number of jobs in the store.
   * 
   * @throws Exception
   *           any exception that occurred during this test (e.g. no service found).
   */
  public void testJobs() throws Exception {
    final PermanentStorage service = getService(PermanentStorage.class);
    for (int i = 0; i < NUMBER_OF_RUNS; i++) {
      final AnyMap any = DataFactory.DEFAULT.createAnyMap();
      final String name = "myName" + i;
      any.put("name", name);
      any.put("workflow", "workflow");
      final JobDefinition job = new JobDefinition(any);
      service.addJob(job);
      final Collection<String> names = service.getJobs();
      assertEquals(1, names.size());
      assertEquals(name, names.iterator().next());
      final JobDefinition read = service.getJob(name);
      assertNotNull(read);
      service.removeJob(name);
      final Collection<String> noNames = service.getJobs();
      assertTrue(noNames.isEmpty());
    }
  }

  /** test storing and reading of job run data. */
  public void testJobRunData() throws Exception {
    final String jobName = "testJobRunData";
    final String jobRunId = "TestRun";
    final PermanentStorage service = getService(PermanentStorage.class);
    final AnyMap jobRunData = DataFactory.DEFAULT.createAnyMap();
    jobRunData.put(JobManagerConstants.DATA_JOB_ID, jobRunId);
    jobRunData.put(JobManagerConstants.DATA_JOB_STATE, JobState.SUCCEEDED.name());
    jobRunData.put(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS, "0");
    jobRunData.put(JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS, "0");
    jobRunData.put(JobManagerConstants.DATA_JOB_NO_OF_STARTED_WORKFLOW_RUNS, "42");
    jobRunData.put(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS, "42");
    service.storeJobRun(jobName, jobRunId, jobRunData);
    assertTrue(service.containsJobRun(jobName, jobRunId));
    assertFalse(service.containsJobRun(jobName, "invalidRunId"));
    assertTrue(service.getJobRunIds("invalid").isEmpty());
    final Collection<String> runIds = service.getJobRunIds(jobName);
    assertEquals(1, runIds.size());
    assertTrue(runIds.contains(jobRunId));
    final AnyMap readRunData = service.getJobRunData(jobName, jobRunId);
    assertEquals(jobRunData, readRunData);
    service.deleteJobRunData(jobName, jobRunId);
    assertTrue(service.getJobRunIds(jobName).isEmpty());
    assertNull(service.getJobRunData(jobName, jobRunId));
  }
}
