/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;

/**
 * Test cases for jobs that depend on other jobs to be running.
 */
public class TestJobDependsOnRunningJob extends JobTaskProcessingTestBase {

  private static final String INDEPENDENT_JOB1 = "independentJob1";

  private static final String INDEPENDENT_JOB2 = "independentJob2";

  private static final String DEPENDENT_JOB1 = "dependentJob1";

  private static final String DEPENDENT_JOB2 = "dependentJob2";

  /** {@inheritDoc} */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
    JobRunInfo info = _jobRunDataProvider.getJobRunInfo(INDEPENDENT_JOB1);
    if (info != null) {
      _jobRunEngine.cancelJob(INDEPENDENT_JOB1, info.getId());
    }
    info = _jobRunDataProvider.getJobRunInfo(INDEPENDENT_JOB2);
    if (info != null) {
      _jobRunEngine.cancelJob(INDEPENDENT_JOB2, info.getId());
    }
    info = _jobRunDataProvider.getJobRunInfo(DEPENDENT_JOB1);
    if (info != null) {
      _jobRunEngine.cancelJob(DEPENDENT_JOB1, info.getId());
    }
    info = _jobRunDataProvider.getJobRunInfo(DEPENDENT_JOB2);
    if (info != null) {
      _jobRunEngine.cancelJob(DEPENDENT_JOB2, info.getId());
    }
  }

  /**
   * Starts a job that is not depending on another job.
   */
  public void testStartJobNotDependingOnOtherJob() throws Exception {
    final String runId = _jobRunEngine.startJob(INDEPENDENT_JOB1);
    final JobRunInfo info = _jobRunDataProvider.getJobRunInfo(INDEPENDENT_JOB1);
    assertEquals(runId, info.getId());
    _jobRunEngine.finishJob(INDEPENDENT_JOB1, runId);
  }

  /**
   * Starts a job that is depending on another job.
   */
  public void testStartJobDependingOnOtherJob() throws Exception {
    String runIdDependentJob;

    try {
      _jobRunEngine.startJob(DEPENDENT_JOB1);
      fail(DEPENDENT_JOB1 + " must not be started. Its referred job is not running, yet.");
    } catch (final JobManagerException e) {
      assertTrue(e.getMessage().contains(INDEPENDENT_JOB1));
    }

    final String runIdIndependentJob = _jobRunEngine.startJob(INDEPENDENT_JOB1);
    final JobRunInfo infoIndependent = _jobRunDataProvider.getJobRunInfo(INDEPENDENT_JOB1);
    assertEquals(runIdIndependentJob, infoIndependent.getId());

    runIdDependentJob = _jobRunEngine.startJob(DEPENDENT_JOB1);
    final JobRunInfo infoDependent = _jobRunDataProvider.getJobRunInfo(DEPENDENT_JOB1);
    assertEquals(runIdDependentJob, infoDependent.getId());

    _jobRunEngine.finishJob(DEPENDENT_JOB1, runIdDependentJob);
    _jobRunEngine.finishJob(INDEPENDENT_JOB1, runIdIndependentJob);
  }

  /**
   * Starts a job that is depending on another job.
   */
  public void testStartJobDependingOnTwoOtherJobs() throws Exception {
    String runIdDependentJob;

    try {
      _jobRunEngine.startJob(DEPENDENT_JOB2);
      fail(DEPENDENT_JOB2 + " must not be started. Its referred job is not running, yet.");
    } catch (final JobManagerException e) {
      assertTrue(e.getMessage().contains(INDEPENDENT_JOB1));
      assertTrue(e.getMessage().contains(INDEPENDENT_JOB2));
    }

    final String runIdIndependentJob = _jobRunEngine.startJob(INDEPENDENT_JOB1);
    final JobRunInfo infoIndependent = _jobRunDataProvider.getJobRunInfo(INDEPENDENT_JOB1);
    assertEquals(runIdIndependentJob, infoIndependent.getId());

    try {
      _jobRunEngine.startJob(DEPENDENT_JOB2);
      fail(DEPENDENT_JOB2 + " must not be started. Its referred job is not running, yet.");
    } catch (final JobManagerException e) {
      assertTrue(e.getMessage().contains(INDEPENDENT_JOB2));
      assertFalse(e.getMessage().contains(INDEPENDENT_JOB1));
    }

    final String runIdIndependentJob2 = _jobRunEngine.startJob(INDEPENDENT_JOB2);
    final JobRunInfo infoIndependent2 = _jobRunDataProvider.getJobRunInfo(INDEPENDENT_JOB2);
    assertEquals(runIdIndependentJob2, infoIndependent2.getId());

    runIdDependentJob = _jobRunEngine.startJob(DEPENDENT_JOB2);
    final JobRunInfo infoDependent = _jobRunDataProvider.getJobRunInfo(DEPENDENT_JOB2);
    assertEquals(runIdDependentJob, infoDependent.getId());

    _jobRunEngine.finishJob(DEPENDENT_JOB2, runIdDependentJob);
    _jobRunEngine.finishJob(INDEPENDENT_JOB1, runIdIndependentJob);
    _jobRunEngine.finishJob(INDEPENDENT_JOB2, runIdIndependentJob2);
  }

  /** Tests stopping of a job that another depends upon. */
  public void testStopDependentJob() throws Exception {
    String runIdDependentJob;

    final String runIdIndependentJob = _jobRunEngine.startJob(INDEPENDENT_JOB1);
    final JobRunInfo infoIndependent = _jobRunDataProvider.getJobRunInfo(INDEPENDENT_JOB1);
    assertEquals(runIdIndependentJob, infoIndependent.getId());

    runIdDependentJob = _jobRunEngine.startJob(DEPENDENT_JOB1);
    final JobRunInfo infoDependent = _jobRunDataProvider.getJobRunInfo(DEPENDENT_JOB1);
    assertEquals(runIdDependentJob, infoDependent.getId());

    try {
      _jobRunEngine.finishJob(INDEPENDENT_JOB1, runIdIndependentJob);
      fail(DEPENDENT_JOB1 + " must not be finished. Its dependent job is still running, yet.");
    } catch (final JobManagerException e) {
      assertTrue(e.getMessage().contains(DEPENDENT_JOB1));
    }

    _jobRunEngine.finishJob(DEPENDENT_JOB1, runIdDependentJob);
    _jobRunEngine.finishJob(INDEPENDENT_JOB1, runIdIndependentJob);
  }

  /** Tests stopping of one job of two jobs that another depends upon. */
  public void testStopDependentJobOfTwoJobs() throws Exception {
    final String runIdIndependentJob1 = _jobRunEngine.startJob(INDEPENDENT_JOB1);
    final String runIdIndependentJob2 = _jobRunEngine.startJob(INDEPENDENT_JOB2);

    final String runIdDependentJob1 = _jobRunEngine.startJob(DEPENDENT_JOB1);
    final String runIdDependentJob2 = _jobRunEngine.startJob(DEPENDENT_JOB2);
    final JobRunInfo infoDependent2 = _jobRunDataProvider.getJobRunInfo(DEPENDENT_JOB2);
    assertEquals(runIdDependentJob2, infoDependent2.getId());

    try {
      _jobRunEngine.finishJob(INDEPENDENT_JOB1, runIdIndependentJob1);
      fail(INDEPENDENT_JOB1 + " must not be finished. Its dependent job is still running.");
    } catch (final JobManagerException e) {
      assertTrue(e.getMessage().contains(DEPENDENT_JOB1));
      assertTrue(e.getMessage().contains(DEPENDENT_JOB2));
    }

    try {
      _jobRunEngine.finishJob(INDEPENDENT_JOB2, runIdIndependentJob2);
      fail(INDEPENDENT_JOB2 + " must not be finished. Its dependent job is still running.");
    } catch (final JobManagerException e) {
      assertTrue(e.getMessage().contains(DEPENDENT_JOB2));
      assertFalse(e.getMessage().contains(DEPENDENT_JOB1));
    }

    _jobRunEngine.finishJob(DEPENDENT_JOB1, runIdDependentJob1);

    try {
      _jobRunEngine.finishJob(INDEPENDENT_JOB1, runIdIndependentJob1);
      fail(INDEPENDENT_JOB1 + " must not be finished. Its dependent job is still running.");
    } catch (final JobManagerException e) {
      assertTrue(e.getMessage().contains(DEPENDENT_JOB2));
    }

    _jobRunEngine.finishJob(DEPENDENT_JOB2, runIdDependentJob2);

    _jobRunEngine.finishJob(INDEPENDENT_JOB2, runIdIndependentJob2);
    _jobRunEngine.finishJob(INDEPENDENT_JOB1, runIdIndependentJob1);
  }

  /** Tests cancelling of a job that another depends upon. */
  public void testCancelDependentJob() throws Exception {
    final String runIdIndependentJob1 = _jobRunEngine.startJob(INDEPENDENT_JOB1);
    final String runIdIndependentJob2 = _jobRunEngine.startJob(INDEPENDENT_JOB2);

    final String runIdDependentJob2 = _jobRunEngine.startJob(DEPENDENT_JOB2);
    final JobRunInfo infoDependent2 = _jobRunDataProvider.getJobRunInfo(DEPENDENT_JOB2);
    assertEquals(runIdDependentJob2, infoDependent2.getId());

    _jobRunEngine.cancelJob(INDEPENDENT_JOB1, runIdIndependentJob1);
    _jobRunEngine.finishJob(DEPENDENT_JOB2, runIdDependentJob2);
    _jobRunEngine.finishJob(INDEPENDENT_JOB2, runIdIndependentJob2);
  }
}
