/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.exceptions.JobRunModeNotAllowedException;

/**
 * Tests for Job Run Modes set by definitions.
 */
public class TestJobRunModes extends JobManagerTestBase {

  private static final String TEST_STORE = "test";

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _objectStoreService.ensureStore(TEST_STORE);
  }

  /** start job with workflow requiring runOnce mode. */
  public void testStartRunOnceWorkflow() throws Exception {
    final String jobName = "TestJobRunModes.testStartRunOnceWorkflow";
    final AnyMap parameters = createJobParameters("test", TEST_STORE, TEST_STORE);
    addJob(jobName, "triggeredWorkflowRunOnce", parameters);
    final String jobRunId = startJob(jobName);
    final AnyMap runData = _jobRunDataProvider.getJobRunData(jobName, jobRunId);
    assertEquals("RUNONCE", runData.getStringValue(JobManagerConstants.DATA_JOB_RUN_MODE));
  }

  /** start job requiring runOnce mode. */
  public void testStartRunOnceJob() throws Exception {
    final String jobName = "TestJobRunModes.testStartRunOnceJob";
    final AnyMap parameters = createJobParameters("test", TEST_STORE, TEST_STORE);
    addJob(jobName, "triggeredWorkflow", parameters, "runOnce");
    final String jobRunId = startJob(jobName);
    final AnyMap runData = _jobRunDataProvider.getJobRunData(jobName, jobRunId);
    assertEquals("RUNONCE", runData.getStringValue(JobManagerConstants.DATA_JOB_RUN_MODE));
  }

  /** start job with workflow requiring runOnce mode with runOnce explicitly requested. */
  public void testStartRunOnceWorkflowAsRunOnce() throws Exception {
    final String jobName = "TestJobRunModes.testStartRunOnceWorkflowAsRunOnce";
    final AnyMap parameters = createJobParameters("test", TEST_STORE, TEST_STORE);
    addJob(jobName, "triggeredWorkflowRunOnce", parameters);
    final String jobRunId = startJob(jobName, JobRunMode.RUNONCE);
    final AnyMap runData = _jobRunDataProvider.getJobRunData(jobName, jobRunId);
    assertEquals("RUNONCE", runData.getStringValue(JobManagerConstants.DATA_JOB_RUN_MODE));
  }

  /** start job requiring runOnce mode. */
  public void testStartRunOnceJobRunOnceAsRunOnce() throws Exception {
    final String jobName = "TestJobRunModes.testStartRunOnceJobRunOnceAsRunOnce";
    final AnyMap parameters = createJobParameters("test", TEST_STORE, TEST_STORE);
    addJob(jobName, "triggeredWorkflow", parameters, "runOnce");
    final String jobRunId = startJob(jobName, JobRunMode.RUNONCE);
    final AnyMap runData = _jobRunDataProvider.getJobRunData(jobName, jobRunId);
    assertEquals("RUNONCE", runData.getStringValue(JobManagerConstants.DATA_JOB_RUN_MODE));
  }

  /** start job with workflow requiring runOnce mode in standard mode. */
  public void testDontStartRunOnceWorkflowAsStandard() throws Exception {
    final String jobName = "TestJobRunModes.testDontStartRunOnceWorkflowAsStandard";
    final AnyMap parameters = createJobParameters("test", TEST_STORE, TEST_STORE);
    addJob(jobName, "triggeredWorkflowRunOnce", parameters);
    try {
      _jobRunEngine.startJob(jobName, JobRunMode.STANDARD);
      fail("should not work");
    } catch (final JobManagerException ex) {
      assertTrue(ex.getCause() instanceof JobRunModeNotAllowedException);
    }
  }

  /** reject start job requiring runOnce mode in standard mode. */
  public void testStartRunOnceJobAsStandard() throws Exception {
    final String jobName = "TestJobRunModes.testStartRunOnceJobAsStandard";
    final AnyMap parameters = createJobParameters("test", TEST_STORE, TEST_STORE);
    addJob(jobName, "triggeredWorkflow", parameters, "runOnce");
    try {
      _jobRunEngine.startJob(jobName, JobRunMode.STANDARD);
      fail("should not work");
    } catch (final JobManagerException ex) {
      assertTrue(ex.getCause() instanceof JobRunModeNotAllowedException);
    }
  }

  /** test if job def can restrict explicitly defined workflow modes. */
  public void testJobDefRestrictsWorkflowModes() throws Exception {
    final String jobName = "TestJobRunModes.testJobDefRestrictsWorkflowModes";
    final AnyMap parameters = createJobParameters("test", TEST_STORE, TEST_STORE);
    addJob(jobName, "triggeredWorkflowRunOnceStandard", parameters, "standard");
    try {
      startJob(jobName, JobRunMode.RUNONCE);
      fail("should not work");
    } catch (final JobManagerException ex) {
      assertTrue(ex.getCause() instanceof JobRunModeNotAllowedException);
    }
  }

  /** test if job def can change expliclty defined workflow run mode. */
  public void testJobDefChangesDefaultWorkflowMode() throws Exception {
    final String jobName = "TestJobRunModes.testJobDefChangesDefaultWorkflowMode";
    final AnyMap parameters = createJobParameters("test", TEST_STORE, TEST_STORE);
    addJob(jobName, "triggeredWorkflowRunOnceStandard", parameters, "standard");
    final String jobRunId = startJob(jobName);
    try {
      final AnyMap runData = _jobRunDataProvider.getJobRunData(jobName, jobRunId);
      assertEquals("STANDARD", runData.getStringValue(JobManagerConstants.DATA_JOB_RUN_MODE));
    } finally {
      _jobRunEngine.cancelJob(jobName, jobRunId);
    }
  }

}
