/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.io.IOException;
import java.util.Collection;
import java.util.List;

import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.test.worker.TestParametersWorker;
import org.eclipse.smila.jobmanager.test.worker.TestParametersWorker2;
import org.eclipse.smila.jobmanager.test.worker.TestParametersWorker3;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskworker.input.RecordInput;

/**
 * Test priority of parameters between action, workflow and job parameters. Uses worker-specific and standard
 * parameters.
 */
public class TestParameterPriority extends JobTaskProcessingTestBase {

  /** job name. */
  private static final String JOB_NAME = "testParameterPriorityJob";

  /** worker name for initial worker (startAction worker). */
  private static final String INPUT_WORKER = "inputWorker";

  /** store name. */
  private static final String STORE_NAME = "teststore";

  /** The object store service. */
  private ObjectStoreService _objectStore;

  /** the jobId (to be able to finish it when test is finished. */
  private String _jobId;

  /** ref. to the taskmanager. */
  private TaskManager _taskManager;

  /** to create BON from record. */
  private final IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _objectStore = getService(ObjectStoreService.class);
    _objectStore.ensureStore(STORE_NAME);
    _taskManager = getService(TaskManager.class);
  }

  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
    // job has already been finished.
    // but if something went wrong, we should maybe try to cancel it:
    try {
      _jobRunEngine.cancelJob(JOB_NAME, _jobId);
    } catch (final JobManagerException e) {
      ; // ignore.
    }
  }

  /** test parameter priority. */
  public void test() throws Exception {
    _jobId = _jobRunEngine.startJob(JOB_NAME);

    final Task currentTask = _jobTaskProcessor.getInitialTask(INPUT_WORKER, JOB_NAME);
    assertNotNull(currentTask);
    createOutputObjects(currentTask);
    final ResultDescription resultDescription =
      new ResultDescription(TaskCompletionStatus.SUCCESSFUL, null, null, null);
    _taskManager.finishTask(INPUT_WORKER, currentTask.getTaskId(), resultDescription);

    _jobRunEngine.finishJob(JOB_NAME, _jobId);
    waitForJobRunCompleted(JOB_NAME, _jobId, 120000);

    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(STORE_NAME, "finalBucket");
    assertEquals(1, storeObjects.size());
    final StoreObject object = storeObjects.iterator().next();

    final RecordInput rinput =
      new RecordInput(new BulkInfo("outbulkBucket", STORE_NAME, object.getId()), _objectStoreService);
    try {
      Record record = rinput.getRecord();
      assertTrue(record != null);
      assertEquals(1, record.getMetadata().getSeq(TestParametersWorker.OUTPUT_ATTRIBUTE).size());
      assertEquals("v3", record.getMetadata().getSeq(TestParametersWorker.OUTPUT_ATTRIBUTE).getStringValue(0));
      assertEquals(2, record.getMetadata().getSeq(TestParametersWorker2.OUTPUT_ATTRIBUTE).size());
      assertEquals("v4", record.getMetadata().getSeq(TestParametersWorker2.OUTPUT_ATTRIBUTE).getStringValue(0));
      assertEquals("v2", record.getMetadata().getSeq(TestParametersWorker2.OUTPUT_ATTRIBUTE).getStringValue(1));
      assertEquals(1, record.getMetadata().getSeq(TestParametersWorker3.OUTPUT_ATTRIBUTE).size());
      assertEquals("v1", record.getMetadata().getSeq(TestParametersWorker3.OUTPUT_ATTRIBUTE).getStringValue(0));
    } finally {
      rinput.close();
    }
  }

  /**
   * Writes a pseudo bulk.
   *
   * @param currentTask
   *          the current task, for which the output buckets are to be filled.
   */
  private void createOutputObjects(final Task currentTask) throws ObjectStoreException, IOException {
    int i = 0;
    for (final List<BulkInfo> bulkInfoList : currentTask.getOutputBulks().values()) {
      for (final BulkInfo bulkInfo : bulkInfoList) {
        final Record record = DataFactory.DEFAULT.createRecord(String.valueOf(i++));
        record.getMetadata().put("dummy", "dummy value");
        _objectStore.putObject(bulkInfo.getStoreName(), bulkInfo.getObjectName(),
          _ipcRecordWriter.writeBinaryObject(record));
      }
    }
  }
}
