/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.Date;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.jobmanager.internal.TaskParameterUtils;

/**
 * Testcases for {@link TaskParameterUtils}.
 */
public class TestTaskParameterUtils extends TestCase {

  /** test {@link TaskParameterUtils#evaluateParameters(AnyMap}. */
  public void testEvaluate() {
    final String stringStr = "string";
    final String doubleStr = "double";
    final String longStr = "long";
    final String boolStr = "bool";
    final String dateStr = "date";
    final String dateTimeStr = "dateTime";
    final String mapStr = "map";
    final String seqStr = "seq";
    final AnyMap param = DataFactory.DEFAULT.createAnyMap();
    final AnyMap expectedResult = DataFactory.DEFAULT.createAnyMap();
    final Value date = DataFactory.DEFAULT.createDateValue(new Date());
    final Value dateTime = DataFactory.DEFAULT.createDateTimeValue(new Date());
    param.put(stringStr, stringStr);
    param.put(longStr, Long.MAX_VALUE);
    param.put(boolStr, true);
    param.put(doubleStr, Double.MIN_VALUE);
    param.put(dateStr, date);
    param.put(dateTimeStr, dateTime);
    expectedResult.put(stringStr, stringStr);
    expectedResult.put(longStr, Long.MAX_VALUE);
    expectedResult.put(boolStr, true);
    expectedResult.put(doubleStr, Double.MIN_VALUE);
    expectedResult.put(dateStr, date);
    expectedResult.put(dateTimeStr, dateTime);
    final AnyMap mapWithinMap = DataFactory.DEFAULT.createAnyMap();
    final AnyMap mapWithinMapWithinMap = DataFactory.DEFAULT.createAnyMap();
    mapWithinMapWithinMap.put(stringStr + mapStr, "${" + stringStr + "}");
    mapWithinMapWithinMap.put(longStr + mapStr, "${" + longStr + "}");
    mapWithinMapWithinMap.put(boolStr + mapStr, "${" + boolStr + "}");
    mapWithinMapWithinMap.put(doubleStr + mapStr, "${" + doubleStr + "}");
    mapWithinMapWithinMap.put(dateStr + mapStr, "${" + dateStr + "}");
    mapWithinMapWithinMap.put(dateTimeStr + mapStr, "${" + dateTimeStr + "}");
    mapWithinMap.put(mapStr, mapWithinMapWithinMap);
    final AnySeq seq = DataFactory.DEFAULT.createAnySeq();
    seq.add("${" + dateStr + "}");
    seq.add("${" + dateTimeStr + "}");
    seq.add("${" + stringStr + "}");
    seq.add("${" + longStr + "}");
    seq.add("${" + doubleStr + "}");
    seq.add("${" + boolStr + "}");
    seq.add(mapWithinMapWithinMap);
    param.put(seqStr, seq);
    param.put(mapStr, mapWithinMap);

    final AnyMap expectedMapWithinMap = DataFactory.DEFAULT.createAnyMap();
    final AnyMap expectedMapWithinMapWithinMap = DataFactory.DEFAULT.createAnyMap();
    expectedMapWithinMapWithinMap.put(stringStr + mapStr, stringStr);
    expectedMapWithinMapWithinMap.put(longStr + mapStr, Long.MAX_VALUE);
    expectedMapWithinMapWithinMap.put(boolStr + mapStr, true);
    expectedMapWithinMapWithinMap.put(doubleStr + mapStr, Double.MIN_VALUE);
    expectedMapWithinMapWithinMap.put(dateStr + mapStr, date);
    expectedMapWithinMapWithinMap.put(dateTimeStr + mapStr, dateTime);
    expectedMapWithinMap.put(mapStr, expectedMapWithinMapWithinMap);
    final AnySeq expectedSeq = DataFactory.DEFAULT.createAnySeq();
    expectedSeq.add(date);
    expectedSeq.add(dateTime);
    expectedSeq.add(stringStr);
    expectedSeq.add(Long.MAX_VALUE);
    expectedSeq.add(Double.MIN_VALUE);
    expectedSeq.add(DataFactory.DEFAULT.createBooleanValue(Boolean.TRUE));
    expectedSeq.add(expectedMapWithinMapWithinMap);
    expectedResult.put(seqStr, expectedSeq);
    expectedResult.put(mapStr, expectedMapWithinMap);

    final AnyMap paramsBefore = DataFactory.DEFAULT.cloneAnyMap(param);
    final AnyMap evaluatedParams = TaskParameterUtils.evaluateParameters(param);
    assertNotNull(evaluatedParams);
    assertEquals(paramsBefore, param);
    // we cannot use equals, since expressions will be converted to strings...
    // so just let's compare the strings created by both maps
    assertEquals(expectedResult.toString(), evaluatedParams.toString());
  }

  /** test {@link TaskParameterUtils#evaluateParameters(AnyMap, prefix)} with non-null prefix. */
  public void testMergeAndEvaluateOneMap() {
    final AnyMap param = DataFactory.DEFAULT.createAnyMap();
    param.put("prefix.expression", "${value}");
    param.put("prefix.value", "1234");
    param.put("other-prefix.expression", "${other-value}");
    param.put("other-prefix.value", "5678");
    param.put("expression", "${no-value}");
    param.put("value", "abc");
    param.put("other-value", "xyz");

    AnyMap evaluatedParams = TaskParameterUtils.mergeAndEvaluateParameters(param, null, null, "prefix");
    assertEquals("1234", evaluatedParams.getStringValue("expression"));
    assertEquals("1234", evaluatedParams.getStringValue("value"));
    assertEquals("1234", evaluatedParams.getStringValue("prefix.expression"));
    assertEquals("xyz", evaluatedParams.getStringValue("other-prefix.expression"));

    evaluatedParams = TaskParameterUtils.mergeAndEvaluateParameters(param, null, null, "other-prefix");
    assertEquals("xyz", evaluatedParams.getStringValue("expression"));
    assertEquals("5678", evaluatedParams.getStringValue("value"));
    assertEquals("5678", evaluatedParams.getStringValue("prefix.expression"));
    assertEquals("xyz", evaluatedParams.getStringValue("other-prefix.expression"));

    evaluatedParams = TaskParameterUtils.mergeAndEvaluateParameters(null, param, null, "prefix");
    assertEquals("1234", evaluatedParams.getStringValue("expression"));
    assertEquals("1234", evaluatedParams.getStringValue("value"));
    assertEquals("1234", evaluatedParams.getStringValue("prefix.expression"));
    assertEquals("xyz", evaluatedParams.getStringValue("other-prefix.expression"));

    evaluatedParams = TaskParameterUtils.mergeAndEvaluateParameters(null, param, null, "other-prefix");
    assertEquals("xyz", evaluatedParams.getStringValue("expression"));
    assertEquals("5678", evaluatedParams.getStringValue("value"));
    assertEquals("5678", evaluatedParams.getStringValue("prefix.expression"));
    assertEquals("xyz", evaluatedParams.getStringValue("other-prefix.expression"));

    // no de-prefexing is done for action parameters
    evaluatedParams = TaskParameterUtils.mergeAndEvaluateParameters(null, null, param, "prefix");
    assertEquals("${no-value}", evaluatedParams.getStringValue("expression"));
    assertEquals("abc", evaluatedParams.getStringValue("value"));
    assertEquals("abc", evaluatedParams.getStringValue("prefix.expression"));
    assertEquals("xyz", evaluatedParams.getStringValue("other-prefix.expression"));
  }

  public void testSimpleParameter() {
    final AnyMap jobParam = DataFactory.DEFAULT.createAnyMap();
    jobParam.put("name1", "v1");
    jobParam.put("worker2.name1", "v2");

    final AnyMap worker1Params = TaskParameterUtils.mergeAndEvaluateParameters(jobParam, null, null, "worker1");
    assertEquals("v1", worker1Params.getStringValue("name1"));

    final AnyMap worker2Params = TaskParameterUtils.mergeAndEvaluateParameters(jobParam, null, null, "worker2");
    assertEquals("v2", worker2Params.getStringValue("name1"));
  }

  public void testComplexParameter() {
    final AnyMap jobParam = DataFactory.DEFAULT.createAnyMap();
    jobParam.getMap("name1", true).put("k1", "v1");
    jobParam.getMap("name1").put("k2", "v2");
    jobParam.getMap("worker2.name1", true).put("k1", "v3");

    final AnyMap worker1Params = TaskParameterUtils.mergeAndEvaluateParameters(jobParam, null, null, "worker1");
    assertEquals("v1", worker1Params.getMap("name1").getStringValue("k1"));
    assertEquals("v2", worker1Params.getMap("name1").getStringValue("k2"));

    final AnyMap worker2Params = TaskParameterUtils.mergeAndEvaluateParameters(jobParam, null, null, "worker2");
    assertEquals("v3", worker2Params.getMap("name1").getStringValue("k1"));
    assertNull(worker2Params.getMap("name1").getStringValue("k2"));
  }

  public void testParameterPriorities() {
    final AnyMap jobParam = DataFactory.DEFAULT.createAnyMap();
    jobParam.put("name1", "v1");
    jobParam.put("worker2.name1", "v2");

    final AnyMap workflowParam = DataFactory.DEFAULT.createAnyMap();
    workflowParam.put("worker1.name1", "v3");

    final AnyMap actionParam = DataFactory.DEFAULT.createAnyMap();
    actionParam.put("name1", "v4");

    final AnyMap worker1Params =
      TaskParameterUtils.mergeAndEvaluateParameters(jobParam, workflowParam, null, "worker1");
    assertEquals("v3", worker1Params.getStringValue("name1"));

    final AnyMap worker2WithActionParams =
      TaskParameterUtils.mergeAndEvaluateParameters(jobParam, workflowParam, actionParam, "worker2");
    assertEquals("v4", worker2WithActionParams.getStringValue("name1"));

    final AnyMap worker2WithoutActionParams =
      TaskParameterUtils.mergeAndEvaluateParameters(jobParam, workflowParam, null, "worker2");
    assertEquals("v2", worker2WithoutActionParams.getStringValue("name1"));

    final AnyMap worker3Params =
      TaskParameterUtils.mergeAndEvaluateParameters(jobParam, workflowParam, null, "worker3");
    assertEquals("v1", worker3Params.getStringValue("name1"));
  }

}
