/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Map;

import junit.framework.TestCase;

import org.eclipse.smila.common.definitions.ValueExpression;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;

/**
 * Testcases for ValueExpression.
 */
public class TestValueExpression extends TestCase {

  /** . */
  private static final String EXPRESSION_1 = "expression1";

  /** . */
  private static final String EXPRESSION_2 = "${variable1}";

  /** . */
  private static final String EXPRESSION_2A = "static/${variable1}";

  /** . */
  private static final String EXPRESSION_3 = "${no-variable";

  /** . */
  private static final String EXPRESSION_4 = "${variable1}blabla${variable1.2}blabla${novariable";

  /** this is possible, but not exactly good style. :-) */
  private static final String EXPRESSION_5 = "${${variable2}}";

  /** value expression tests conversion to and from string. */
  public void testValueExpressionFromAndToString() {
    final ValueExpression ve = new ValueExpression(EXPRESSION_1);
    assertEquals(ve.getExpression(), EXPRESSION_1);
    final ValueExpression veIdentical = new ValueExpression(EXPRESSION_1);
    assertEquals("equals() does not work correctly.", ve, veIdentical);

    final ValueExpression ve2 = new ValueExpression(EXPRESSION_2);
    assertEquals(ve2.getExpression(), EXPRESSION_2);
    final ValueExpression ve3 = new ValueExpression(EXPRESSION_3);
    assertEquals(ve3.getExpression(), EXPRESSION_3);
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    assertEquals(ve4.getExpression(), EXPRESSION_4);
    final ValueExpression ve5 = new ValueExpression(EXPRESSION_5);
    assertEquals(ve5.getExpression(), EXPRESSION_5);
  }

  /** test referencesVariables(). */
  public void testValueExpressionReferencesVariables() {
    final ValueExpression ve1 = new ValueExpression(EXPRESSION_1);
    assertFalse(ve1.referencesVariables());
    final ValueExpression ve2 = new ValueExpression(EXPRESSION_2);
    assertTrue(ve2.referencesVariables());
    final ValueExpression ve3 = new ValueExpression(EXPRESSION_3);
    assertFalse(ve3.referencesVariables());
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    assertTrue(ve4.referencesVariables());
    final ValueExpression ve5 = new ValueExpression(EXPRESSION_5);
    assertTrue(ve5.referencesVariables());
  }

  /** test getVariables(). */
  public void testValueExpressionGetReferencedVariableNames() {
    final ValueExpression ve1 = new ValueExpression(EXPRESSION_1);
    final Collection<String> ref1 = ve1.getVariables();
    assertTrue(ref1.size() == 0);
    final ValueExpression ve2 = new ValueExpression(EXPRESSION_2);
    final Collection<String> ref2 = ve2.getVariables();
    assertTrue(ref2.size() == 1);
    assertTrue(ref2.contains("variable1"));
    final ValueExpression ve3 = new ValueExpression(EXPRESSION_3);
    final Collection<String> ref3 = ve3.getVariables();
    assertTrue(ref3.size() == 0);
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    final Collection<String> ref4 = ve4.getVariables();
    assertTrue(ref4.size() == 2);
    assertTrue(ref4.contains("variable1"));
    assertTrue(ref4.contains("variable1.2"));
    final ValueExpression ve5 = new ValueExpression(EXPRESSION_5);
    final Collection<String> ref5 = ve5.getVariables();
    assertTrue(ref5.size() == 1);
    assertTrue(ref5.contains("variable2"));
    final Map<String, String> context = new LinkedHashMap<String, String>();
    context.put("variable2", "newVariable");
    final Collection<String> ref6 = ve5.evaluate(context).getVariables();
    assertTrue(ref6.size() == 1);
    assertTrue(ref6.contains("newVariable"));
  }

  /** test getExpression(). */
  public void testValueExpressionEvaluate() {
    final AnyMap context = DataFactory.DEFAULT.createAnyMap();
    context.put("variable1", "value1");
    context.put("variable1.2", "value2");
    final ValueExpression ve1 = new ValueExpression(EXPRESSION_1);
    assertEquals(ve1.evaluate(context).getExpression(), EXPRESSION_1);
    final ValueExpression ve2 = new ValueExpression(EXPRESSION_2);
    assertEquals(ve2.evaluate(context).getExpression(), "value1");
    final ValueExpression ve3 = new ValueExpression(EXPRESSION_3);
    assertEquals(ve3.evaluate(context).getExpression(), EXPRESSION_3);
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    assertEquals(ve4.evaluate(context).getExpression(), "value1blablavalue2blabla${novariable");
  }

  /** test getExpression() with variables referencing another variable. */
  public void testValueExpressionEvaluateWithVariablesReferencingAnotherVariable() {
    final AnyMap context = DataFactory.DEFAULT.createAnyMap();
    context.put("variable1", "value1");
    context.put("variable1.2", "${variable1}-${variable2}");
    context.put("variable2", "value2");
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    assertEquals(ve4.evaluate(context).getExpression(), "value1blablavalue1-value2blabla${novariable");
  }

  /** test getExpression() with variables referencing a map entry. */
  public void testValueExpressionEvaluateWithVariablesReferencingAMap() {
    final AnyMap context = DataFactory.DEFAULT.createAnyMap();
    context.put("variable1", "value1");
    final AnyMap mapWithin = DataFactory.DEFAULT.createAnyMap();
    mapWithin.put("variable1.2", "${variable1}-${variable2}");
    context.put("map-in-map", mapWithin);
    context.put("variable2", "value2");
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    // be sure that only the top-level variables are resolved
    assertEquals(ve4.evaluate(context).getExpression(), "value1blabla${variable1.2}blabla${novariable");
  }

  /** test getExpression() with a variable referencing a long. */
  public void testValueExpressionEvaluateWithVariablesReferencingALongValue() {
    final AnyMap context = DataFactory.DEFAULT.createAnyMap();
    context.put("variable1", "value1");
    context.put("variable1.2", 42l);
    context.put("variable2", "value2");
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    // be sure that only the top-level variables are resolved
    assertEquals(ve4.evaluate(context).getExpression(), "value1blabla42blabla${novariable");
  }

  /** test getExpression() with a variable referencing a boolean. */
  public void testValueExpressionEvaluateWithVariablesReferencingABooleanValue() {
    final AnyMap context = DataFactory.DEFAULT.createAnyMap();
    context.put("variable1", "value1");
    context.put("variable1.2", true);
    context.put("variable2", "value2");
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    // be sure that only the top-level variables are resolved
    assertEquals(ve4.evaluate(context).getExpression(), "value1blablatrueblabla${novariable");
  }

  /** test getExpression() with a variable referencing a double. */
  public void testValueExpressionEvaluateWithVariablesReferencingADoubleValue() {
    final AnyMap context = DataFactory.DEFAULT.createAnyMap();
    context.put("variable1", "value1");
    context.put("variable1.2", 1.0);
    context.put("variable2", "value2");
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    // be sure that only the top-level variables are resolved
    assertEquals(ve4.evaluate(context).getExpression(), "value1blabla1.0blabla${novariable");
  }

  /** test getExpression() with a variable referencing a date. */
  public void testValueExpressionEvaluateWithVariablesReferencingADateValue() {
    final AnyMap context = DataFactory.DEFAULT.createAnyMap();
    final Date value = new Date();
    context.put("variable1", "value1");
    context.put("variable1.2", DataFactory.DEFAULT.createDateValue(value));
    context.put("variable2", "value2");
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    // be sure that only the top-level variables are resolved
    assertEquals(ve4.evaluate(context).getExpression(), "value1blabla"
      + DataFactory.DEFAULT.createDateValue(value).asString() + "blabla${novariable");
  }

  /** test getExpression() with a variable referencing a date time. */
  public void testValueExpressionEvaluateWithVariablesReferencingADateTimeValue() {
    final AnyMap context = DataFactory.DEFAULT.createAnyMap();
    final Date value = new Date();
    context.put("variable1", "value1");
    context.put("variable1.2", DataFactory.DEFAULT.createDateTimeValue(value));
    context.put("variable2", "value2");
    final ValueExpression ve4 = new ValueExpression(EXPRESSION_4);
    // be sure that only the top-level variables are resolved
    assertEquals(ve4.evaluate(context).getExpression(),
      "value1blabla" + DataFactory.DEFAULT.createDateTimeValue(value).asString() + "blabla${novariable");
  }

  /**
   * test evaluate with variable indirections.
   */
  public void testValueExpressionWithVariableIndirection() {
    final AnyMap context = DataFactory.DEFAULT.createAnyMap();
    context.put("variable1", "value1");
    context.put("variable1.2", "value2");
    context.put("variable2", "variable1");
    final ValueExpression ve5 = new ValueExpression(EXPRESSION_5);
    assertEquals(ve5.evaluate(context).getExpression(), "value1");
  }

  /** tests variable extraction where there are no variables. */
  public void testExtractValuesWithoutVariables() throws Exception {
    final ValueExpression ve1 = new ValueExpression(EXPRESSION_1);
    assertEquals(0, ve1.extractValues("expression1").size());
    try {
      ve1.extractValues("no-match");
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception: " + ex.toString(), ex instanceof IllegalArgumentException);
    }
  }

  /** extract values where there is only one variable. */
  public void testExtractValuesWithVariableOnly() throws Exception {
    final ValueExpression ve1 = new ValueExpression(EXPRESSION_2);
    Map<String, String> values = ve1.extractValues("value1");
    assertEquals(1, values.size());
    assertEquals("value1", values.get("variable1"));
    values = ve1.extractValues("");
    assertEquals(1, values.size());
    assertEquals("", values.get("variable1"));
  }

  /** extract values where there is one text and one variable. */
  public void testExtractValuesWithTextAndVariable() throws Exception {
    final ValueExpression ve1 = new ValueExpression(EXPRESSION_2A);
    Map<String, String> values = ve1.extractValues("static/value1");
    assertEquals(1, values.size());
    assertEquals("value1", values.get("variable1"));
    values = ve1.extractValues("static/abcd-1234-e5f6g7h8");
    assertEquals(1, values.size());
    assertEquals("abcd-1234-e5f6g7h8", values.get("variable1"));
    try {
      ve1.extractValues("no-match/value1");
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception: " + ex.toString(), ex instanceof IllegalArgumentException);
    }
  }

  /** extract values where there are one text and two variables. */
  public void testExtractValuesWithTextAndTwoVariables() throws Exception {
    final ValueExpression ve1 = new ValueExpression(EXPRESSION_4);
    final Map<String, String> values = ve1.extractValues("value1blablavalue1.2blabla${novariable");
    assertEquals(2, values.size());
    assertEquals("value1", values.get("variable1"));
    assertEquals("value1.2", values.get("variable1.2"));
    try {
      ve1.extractValues("value1blablavalue1.2blablavalue3");
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception: " + ex.toString(), ex instanceof IllegalArgumentException);
    }
  }
}
