/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Marco Strack (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.io.IOException;
import java.util.List;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskmanager.TaskManager;

/**
 * Tests functionality of qualifying parameters by workername.
 */
public class TestWorkerSpecificParameters extends JobTaskProcessingTestBase {

  /** worker name for initial worker (startAction worker). */
  private static final String FIRST_WORKER = "paramWorker1";

  /** workflow name. */
  private static final String WORKFLOW_NAME = "testWorkerSpecificParametersWorkflow";

  /** job name. */
  private static final String JOB_NAME = "testWorkerSpecificParametersJob";

  /** store name. */
  private static final String STORE_NAME = "test";

  /** The object store service. */
  private ObjectStoreService _defStorage;

  /** the jobId (to be able to finish it when test is finished. */
  private String _jobId;

  /** ref. to the taskmanager. */
  private TaskManager _taskManager;

  /** to create BON from record. */
  private final IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  private void setUpJob() throws Exception {

    // create a job
    /** the job parameters. */
    final AnyMap _parameters = DataFactory.DEFAULT.createAnyMap();
    final AnyMap jobAny = AccessAny.FACTORY.createAnyMap();
    jobAny.put("name", JOB_NAME);
    jobAny.put("workflow", WORKFLOW_NAME);

    //worker #1
    _parameters.put("store", STORE_NAME);

    //mandatory fields
    _parameters.put("stringM", "marcoM");
    //set this with worker prefix
    _parameters.put(FIRST_WORKER + ".booleanParam", true);

    //test for job param validation
    final AnyMap anyParamM1 = AccessAny.FACTORY.createAnyMap();
    anyParamM1.put("key01", "value01");
    anyParamM1.put("key02", true);
    _parameters.put("anyParamM", anyParamM1);
    final AnyMap anyParamM2 = AccessAny.FACTORY.createAnyMap();
    anyParamM2.put("key01", "value02");
    anyParamM2.put("key02", false);
    _parameters.put(FIRST_WORKER + ".anyParamM", anyParamM2);

    //optional fields
    _parameters.put("stringO", "marcoO1");
    //overwrite with this
    _parameters.put(FIRST_WORKER + ".stringO", "marcoO2");

    jobAny.put("parameters", _parameters);

    _defPersistence.addJob(new JobDefinition(jobAny));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _defStorage = getService(ObjectStoreService.class);
    for (final String storeName : _objectStoreService.getStoreNames()) {
      if ("jobmanager".equals(storeName)) {
        _objectStoreService.clearStore(storeName);
      } else {
        _objectStoreService.removeStore(storeName);
      }
    }
    _defStorage.ensureStore(STORE_NAME);

    _taskManager = getService(TaskManager.class);

    setUpJob();

    _jobId = _jobRunEngine.startJob(JOB_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
    // job has already been finished.
    // but if something went wrong, we should maybe try to cancel it:
    try {
      _jobRunEngine.cancelJob(JOB_NAME, _jobId);
    } catch (final JobManagerException e) {
      ; // ignore.
    }
    _defPersistence.removeJob(JOB_NAME);
  }

  /**
   * tests evaluation and access on worker parameters in job processing.
   *
   * @throws Exception
   *           something went wrong.
   *
   */
  public void testEvaluatedParametersForWorker() throws Exception {
    final Task currentTask = _jobTaskProcessor.getInitialTask(FIRST_WORKER, JOB_NAME);
    assertNotNull(currentTask);
    //check basic set
    //regulars
    assertTrue(currentTask.getParameters().containsKey("stringM"));
    assertEquals("marcoM", currentTask.getParameters().getStringValue("stringM"));
    //from worker specific fields
    assertTrue(currentTask.getParameters().containsKey("booleanParam"));
    assertEquals(true, currentTask.getParameters().getBooleanValue("booleanParam").booleanValue());

    assertTrue(currentTask.getParameters().containsKey("anyParamM"));
    final AnyMap anyParamM = currentTask.getParameters().get("anyParamM").asMap();
    assertTrue(anyParamM.containsKey("key01"));
    assertTrue(anyParamM.containsKey("key02"));
    assertEquals("value02", anyParamM.getStringValue("key01"));
    assertTrue(!anyParamM.getBooleanValue("key02"));

    //check overwrite
    assertTrue(currentTask.getParameters().containsKey("stringO"));
    assertEquals("marcoO2", currentTask.getParameters().getStringValue("stringO"));

    //check workflow defaults
    assertTrue(currentTask.getParameters().containsKey("doubleO"));
    assertEquals(2.5, currentTask.getParameters().getDoubleValue("doubleO").doubleValue());

    final ResultDescription resultDescription =
      new ResultDescription(TaskCompletionStatus.SUCCESSFUL, null, null, null);
    createOutputObjects(currentTask);
    _taskManager.finishTask(FIRST_WORKER, currentTask.getTaskId(), resultDescription);
    _jobRunEngine.finishJob(JOB_NAME, _jobId);
    waitForJobRunCompleted(JOB_NAME, _jobId, 120000);
  }

  /**
   * Writes an pseudo bulk.
   *
   * @param currentTask
   *          the current task, for which the output buckets are to be filled.
   * @throws ObjectStoreException
   *           creation of dummy data failed.
   * @throws IOException
   *           cannot convert record to bon.
   */
  private void createOutputObjects(final Task currentTask) throws ObjectStoreException, IOException {
    int i = 0;
    for (final List<BulkInfo> bulkInfoList : currentTask.getOutputBulks().values()) {
      for (final BulkInfo bulkInfo : bulkInfoList) {
        final Record record = DataFactory.DEFAULT.createRecord(String.valueOf(i++));
        record.getMetadata().put("dummy", "dummy value");
        _defStorage.putObject(bulkInfo.getStoreName(), bulkInfo.getObjectName(),
          _ipcRecordWriter.writeBinaryObject(record));
      }
    }
  }
}
